package ru.yandex.chemodan.app.telemost.web.v2.actions;

import java.util.UUID;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.telemost.chat.model.Chat;
import ru.yandex.chemodan.app.telemost.chat.model.ChatHistory;
import ru.yandex.chemodan.app.telemost.services.ChatService;
import ru.yandex.chemodan.app.telemost.services.model.ChatType;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.web.v2.model.ChatData;
import ru.yandex.chemodan.app.telemost.web.v2.model.ChatHistoryData;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ActionContainer
@AllArgsConstructor
public class ChatsActionsV2 {
    private final ChatService chatService;

    private static final Logger logger = LoggerFactory.getLogger(ChatsActionsV2.class);

    @Path(value = "/v2/create_chat/{room_id}", methods = {HttpMethod.PUT})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ChatData createChat(@PathParam("room_id") String roomId,
                               @RequestParam(value = "user") String user,
                               @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                               @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket,
                               @SpecialParam ActionInvocationContext invocationContext)
    {
        Option<Chat> chatO = Option.empty();
        try {
            chatO = chatService.createChat(roomId, ChatType.CONFERENCE, UUID.fromString(user), uid, tvmUserTicket);
        } catch (Exception e) {
            logger.info("Create chat: error on create chat for conference={}, uid={}, user={}, error={}, stacktrace={}",
                    roomId, uid, user, e.toString(), ExceptionUtils.getStackTrace(e));
        }
        if (!chatO.isPresent()) {
            invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_503_SERVICE_UNAVAILABLE);
            return new ChatData("");
        }
        invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_200_OK);
        logger.info("Create chat: chat created successfully for conference={}, uid={}, user={}, chat_path={}",
                roomId, uid, user, chatO.get().getChatPath());
        return new ChatData(chatO.get());
    }

    @Path(value = "/v2/join_to_chat/{room_id}", methods = {HttpMethod.PUT})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ChatData joinToChat(@PathParam("room_id") String roomId,
                               @RequestParam(value = "user") String user,
                               @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                               @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket,
                               @SpecialParam ActionInvocationContext invocationContext)
    {
        // Заложена следующая логика:
        // 1. Если существует чат встречи, то добавляем участника в него
        // 2. Если существует трансляционный чат, то добавляем участника в него
        // 3. Возвращаем информацию о последнем чате, в который добавили участника
        // 4. Если чатов нет или при попытке добавления в чаты возникла ошибка, то возвращаем код 405
        Option<Chat> conferenceChatO = Option.empty();
        Option<Chat> broadcastChatO = Option.empty();
        boolean warError = false;

        // Попытка подключения к чату встречи, если он существует (может быть отдана ошибка)
        try {
            conferenceChatO = chatService.joinToChatIfExisting(roomId, ChatType.CONFERENCE, UUID.fromString(user), uid,
                    tvmUserTicket);
        } catch (Exception ignored) {
            warError = true;
        }
        if (conferenceChatO.isPresent() && !warError) {
            logger.info("Join to chat: joined to conference chat successfully for conference={}, type={}, uid={}, user={}, chat_path={}",
                    roomId, ChatType.CONFERENCE, uid, user, conferenceChatO.get().getChatPath());
        }

        // Попытка подключения к чату трансляции, если он существует (может быть отдана ошибка)
        try {
            broadcastChatO = chatService.joinToChatIfExisting(roomId, ChatType.BROADCAST, UUID.fromString(user), uid,
                    tvmUserTicket);
        } catch (Exception ignored) {
            warError = true;
        }
        if (broadcastChatO.isPresent() && !warError) {
            logger.info("Join to chat: joined to broadcast chat successfully for conference={}, type={}, uid={}, user={}, chat_path={}",
                    roomId, ChatType.BROADCAST, uid, user, broadcastChatO.get().getChatPath());
        }

        // Финальная обработка ошибки
        if ((conferenceChatO.isEmpty() && broadcastChatO.isEmpty()) || warError) {
            invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_405_METHOD_NOT_ALLOWED);
            return new ChatData("");
        }

        if (broadcastChatO.isPresent()) {
            return new ChatData(broadcastChatO.get());
        }

        return new ChatData(conferenceChatO.get());
    }

    @Path(value = "/v2/get_chat_history/{room_id}", methods = {HttpMethod.PUT})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ChatHistoryData getChatHistory(@PathParam("room_id") String roomId,
                                          @RequestParam(value = "user") String user,
                                          @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                                          @RequestParam(value = "offset") Option<Long> offset,
                                          @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket,
                                          @SpecialParam ActionInvocationContext invocationContext)
    {
        invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_200_OK);
        logger.info("Get chat history request: conference={}, uid={}, user={}, offset={}",
                roomId, uid, user, offset);
        Option<ChatHistory> chatHistoryO = chatService.getChatHistory(
                roomId, ChatType.CONFERENCE, UUID.fromString(user), uid, offset, tvmUserTicket);
        if (!chatHistoryO.isPresent()) {
            logger.info("Get chat history: is empty for conference={}, uid={}, user={}, offset={}",
                    roomId, uid, user, offset);
            return new ChatHistoryData();
        }
        logger.info("Get chat history: success for conference={}, uid={}, user={}, offset={}",
                roomId, uid, user, offset);
        return new ChatHistoryData(chatHistoryO.get());
    }
}
