package ru.yandex.chemodan.app.telemost.web.v2.actions;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.AdmitCommand;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.Command;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.KickCommand;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.MuteCameraCommand;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.MuteDesktopCommand;
import ru.yandex.chemodan.app.telemost.appmessages.model.commands.MuteMicrophoneCommand;
import ru.yandex.chemodan.app.telemost.exceptions.ConferenceNotFoundTelemostException;
import ru.yandex.chemodan.app.telemost.exceptions.TooMuchDataException;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferencePeerDao;
import ru.yandex.chemodan.app.telemost.repository.dao.ConferenceUserDao;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceDto;
import ru.yandex.chemodan.app.telemost.repository.model.ConferenceUserDto;
import ru.yandex.chemodan.app.telemost.repository.model.UserRole;
import ru.yandex.chemodan.app.telemost.services.ChatService;
import ru.yandex.chemodan.app.telemost.services.ClientConfigurationService;
import ru.yandex.chemodan.app.telemost.services.CommandService;
import ru.yandex.chemodan.app.telemost.services.ConferenceParticipantsService;
import ru.yandex.chemodan.app.telemost.services.ConferencePeerService;
import ru.yandex.chemodan.app.telemost.services.ConferenceService;
import ru.yandex.chemodan.app.telemost.services.LimitsService;
import ru.yandex.chemodan.app.telemost.services.OverloadService;
import ru.yandex.chemodan.app.telemost.services.ParticipantType;
import ru.yandex.chemodan.app.telemost.services.RoomService;
import ru.yandex.chemodan.app.telemost.services.UserService;
import ru.yandex.chemodan.app.telemost.services.model.Conference;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceClientParameters;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceParticipant;
import ru.yandex.chemodan.app.telemost.services.model.ConferenceUriData;
import ru.yandex.chemodan.app.telemost.services.model.PassportOrYaTeamUid;
import ru.yandex.chemodan.app.telemost.services.model.RoomConnectionInfo;
import ru.yandex.chemodan.app.telemost.services.model.User;
import ru.yandex.chemodan.app.telemost.web.v2.model.ConferenceConnectionData;
import ru.yandex.chemodan.app.telemost.web.v2.model.ConferenceData;
import ru.yandex.chemodan.app.telemost.web.v2.model.ConferenceLeaveData;
import ru.yandex.chemodan.app.telemost.web.v2.model.ConferenceParticipantData;
import ru.yandex.chemodan.app.telemost.web.v2.model.ConferenceParticipantsData;
import ru.yandex.chemodan.app.telemost.web.v2.model.PeersRequestData;
import ru.yandex.chemodan.web.EmptyPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.PathParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestHeader;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@ActionContainer
@AllArgsConstructor
public class ConferencesActionsV2 {
    private static final Logger logger = LoggerFactory.getLogger(ConferencesActionsV2.class);

    private final RoomService roomService;
    private final ConferenceParticipantsService conferenceParticipantsService;
    private final ConferenceService conferenceService;
    private final ClientConfigurationService clientConfigurationService;
    private final ConferencePeerService conferencePeerService;
    private final LimitsService limitsService;
    private final ChatService chatService;
    private final CommandService commandService;
    private final UserService userService;

    private final ConferenceUserDao conferenceUserDao;
    private final ConferencePeerDao conferencePeerDao;

    private final OverloadService overloadService;

    private final DynamicProperty<Integer> peersDataLimit =
            new DynamicProperty<>("telemost-peers-data-limit-per-request", 200);

    private void addUserIfNotExists(Option<PassportOrYaTeamUid> uid) {
        uid.ifPresent(userService::addUserIfNotExists);
    }

    private void addUserIfNotExists(PassportOrYaTeamUid uid) {
        userService.addUserIfNotExists(uid);
    }

    @Path(value = "/v2/conferences", methods = {HttpMethod.POST})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceConnectionData createConference(@RequestParam("uid") Option<PassportOrYaTeamUid> uid,
                                                     @RequestParam("staffOnly") Option<Boolean> isStaffOnly,
                                                     @RequestParam("isPermanent") Option<Boolean> isPermanent,
                                                     @RequestParam(value = "lang") Option<String> language,
                                                     @RequestParam("clientInstanceId") Option<String> clientInstanceId,
                                                     @RequestParam("externalMeeting") Option<Boolean> isExternalMeeting,
                                                     @RequestParam("calendarEventId") Option<String> eventId,
                                                     @SpecialParam ActionInvocationContext invocationContext)
    {
        addUserIfNotExists(uid);
        overloadService.checkOverload();

        clientInstanceId = clientInstanceId.filter(StringUtils::isNotBlank);
        Option<User> user = uid.map(conferencePeerService::findUser);
        Conference conference = conferenceService.generateConference(
                ConferenceClientParameters.builder()
                        .user(user)
                        .staffOnly(isStaffOnly)
                        .permanent(isPermanent)
                        .externalMeeting(isExternalMeeting)
                        .eventId(eventId).build()
        );
        RoomConnectionInfo room = roomService.createRoomAndUserForConference(conference, user, language, clientInstanceId);
        MapF<String, Object> clientConfiguration = clientConfigurationService.getV2ClientConfiguration();

        invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_201_CREATED);
        return new ConferenceConnectionData(conference, room, clientConfiguration);
    }

    @Path(value = "/v2/conferences/{uri:.*}/connection")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceConnectionData getConferenceConnection(@PathParam("uri") String uri,
                                                            @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                                                            @RequestParam(value = "displayName") Option<String> displayName,
                                                            @RequestParam(value = "lang") Option<String> language,
                                                            @RequestParam(value = "clientInstanceId") Option<String> clientInstanceId,
                                                            @RequestParam(value = "translatorToken") Option<String> translatorToken,
                                                            @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket)
    {
        addUserIfNotExists(uid);
        overloadService.checkOverload(uri, uid);

        clientInstanceId = clientInstanceId.filter(StringUtils::isNotBlank);
        Option<User> user = uid.map(conferencePeerService::findUser);
        ConferenceUriData conferenceUriData = conferenceService.getConferenceUriId(uri);
        Conference conference = conferenceService.joinConference(user, conferenceUriData, tvmUserTicket, translatorToken);
        RoomConnectionInfo room = roomService.joinToConference(
                conference, user, displayName, language, clientInstanceId,
                translatorToken.isPresent() ? ParticipantType.TRANSLATOR : ParticipantType.USER);

        chatService.tryCreateBroadcastChatIfNeed(conference.getConferenceDto(), uid, tvmUserTicket);

        MapF<String, Object> clientConfiguration = clientConfigurationService.getV2ClientConfiguration();

        return new ConferenceConnectionData(conference, room, clientConfiguration);
    }

    @Path(value = "/v2/conferences/{uri:.*}/link_calendar_event", methods = {HttpMethod.POST})
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceData linkCalendarEvent(@PathParam("uri") String uri,
                                            @RequestParam("uid") PassportOrYaTeamUid uid,
                                            @RequestParam("calendarEventId") String calendarEventId)
    {
        addUserIfNotExists(uid);

        Conference conference = conferenceService.linkCalendarEvent(
                conferenceService.getConferenceUriId(uri), uid, calendarEventId);

        return new ConferenceData(conference);
    }

    @Path(value = "/v2/conferences/{uri:.*}")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public ConferenceData getConference(@PathParam("uri") String uri,
                                        @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                                        @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket)
    {
        addUserIfNotExists(uid);
        overloadService.checkOverload(uri, uid);

        ConferenceUriData conferenceUriData = conferenceService.getConferenceUriId(uri);
        Option<User> user = uid.map(conferencePeerService::findUser);
        Conference conference = conferenceService.findOrCreateConferenceWithoutTokenActivating(
                user, conferenceUriData, tvmUserTicket);

        return new ConferenceData(conference);
    }

    @Path(methods = HttpMethod.PUT, value = "/v2/conferences/{uri:.*}/peers")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceParticipantsData getConferencePeerList(@PathParam("uri") String uri,
                                                            @RequestParam(value = "uid") Option<PassportOrYaTeamUid> uid,
                                                            @RequestHeader(value = TvmHeaders.USER_TICKET, required = false) Option<String> tvmUserTicket,
                                                            @BoundByJackson PeersRequestData peers)
    {
        addUserIfNotExists(uid);

        ConferenceUriData conferenceUriData = conferenceService.getConferenceUriId(uri);
        Option<User> user = uid.map(conferencePeerService::findUser);
        Conference conference = conferenceService.findOrCreateConference(user, conferenceUriData, tvmUserTicket);
        int limitValue = limitsService.getParticipantsLimitValue(conference).orElseGet(peersDataLimit::get);
        if (peers.getPeersIds().size() > limitValue) {
            throw new TooMuchDataException(String.format("Too much peers_ids, list limited to %s values", limitValue));
        }

        ListF<ConferenceParticipantData> participants = Cf.arrayList();
        for (ConferenceParticipant conferenceUser : conferenceParticipantsService.findConferenceUsers(conference,
                peers.getPeersIds())) {
            UserRole role = conferenceUser.getUid().map(u -> conferenceUserDao.findByConferenceAndUid(conference.getDbId(), u))
                    .flatMapO(Function.identityF())
                    .map(ConferenceUserDto::getRole)
                    .orElse(UserRole.MEMBER);
            participants.add(new ConferenceParticipantData(conferenceUser, role));
        }

        return new ConferenceParticipantsData(participants);
    }

    @Path(value = "/v2/rooms/{room_id}/leave", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public ConferenceLeaveData leaveConference(@PathParam("room_id") String roomId,
                                               @RequestParam("peerId") String peerId,
                                               @RequestParam("peerToken") String peerToken,
                                               @RequestParam("mediaSessionId") String mediaSessionId,
                                               @SpecialParam ActionInvocationContext invocationContext)
    {
        Conference conference = conferenceService.findConferenceUnsafe(roomId)
                .getOrThrow(ConferenceNotFoundTelemostException::new);
        roomService.disconnectUserFromConference(conference, peerId, peerToken, mediaSessionId);
        if (conferencePeerDao.findActivePeers(conference.getConferenceId()).isEmpty()) {
            logger.info("Remove empty chat for conference={}: conference is empty", conference.getConferenceId());
            try {
                chatService.checkChatHistoryAndRemoveIfEmpty(conference);
            }
            catch (Exception e) {
                logger.info("Couldn't delete an empty chat:: conference={}", conference.getConferenceId());
            }
        }
        invocationContext.getHttpContext().setStatusCode(HttpStatus.SC_200_OK);
        return new ConferenceLeaveData();
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-camera")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteCamera(@PathParam("uri") String uri,
                                @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                @RequestParam("peerId") String peerId)
    {
        sendCommand(uri, uid, peerId, new MuteCameraCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-microphone")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteMicrophone(@PathParam("uri") String uri,
                                    @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                    @RequestParam("peerId") String peerId)
    {
        sendCommand(uri, uid, peerId, new MuteMicrophoneCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-desktop")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteDesktop(@PathParam("uri") String uri,
                                 @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                 @RequestParam("peerId") String peerId)
    {
        sendCommand(uri, uid, peerId, new MuteDesktopCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-all-cameras")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteAllCameras(@PathParam("uri") String uri,
                                    @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                    @RequestParam("excludePeerId") String excludePeerId)
    {
        sendCommandToAll(uri, uid, excludePeerId, new MuteCameraCommand());
        return EmptyPojo.INSTANCE;
    }


    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-all-microphones")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteAllMicrophones(@PathParam("uri") String uri,
                                        @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                        @RequestParam("excludePeerId") String excludePeerId)
    {
        sendCommandToAll(uri, uid, excludePeerId, new MuteMicrophoneCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/mute-all-desktops")
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SYNC_SM)
    public EmptyPojo muteAllDesktops(@PathParam("uri") String uri,
                                     @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                                     @RequestParam("excludePeerId") String excludePeerId)
    {
        sendCommandToAll(uri, uid, excludePeerId, new MuteDesktopCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/set-user-role")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public EmptyPojo setRole(@PathParam("uri") String uri,
                             @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                             @RequestParam(value = "targetUid") PassportOrYaTeamUid targetUid,
                             @RequestParam(value = "userRole") UserRole newRole)
    {
        ConferenceDto conferenceDto = conferenceService.getConferenceByFullUri(uri);
        commandService.setRole(conferenceDto, uid, targetUid, newRole);
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/admit")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public EmptyPojo admit(@PathParam("uri") String uri,
                           @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                           @RequestParam("peerId") String peerId)
    {
        sendCommandGroupByUid(uri, uid, peerId, new AdmitCommand());
        return EmptyPojo.INSTANCE;
    }

    @Path(methods = HttpMethod.POST, value = "/v2/conferences/{uri:.*}/commands/kick")
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public EmptyPojo kick(@PathParam("uri") String uri,
                          @RequestParam(value = "uid") PassportOrYaTeamUid uid,
                          @RequestParam("peerId") String peerId)
    {
        sendCommandGroupByUid(uri, uid, peerId, new KickCommand());
        return EmptyPojo.INSTANCE;
    }

    private void sendCommand(String uri, PassportOrYaTeamUid uid, String peerId, Command command)
    {
        ConferenceDto conferenceDto = conferenceService.getConferenceByFullUri(uri);
        commandService.sendCommand(conferenceDto, peerId, uid, command);
    }

    private void sendCommandGroupByUid(String uri, PassportOrYaTeamUid uid, String peerId, Command command)
    {
        ConferenceDto conferenceDto = conferenceService.getConferenceByFullUri(uri);
        // TODO Для стенда команду отправляем только указанному пиру. В целевой модели нужно заменить на отправку
        //      команды всем пирам, привязанным к uid указанного пира.
        //      Также требуется в отправку команды добавить данные админа, который ее вызвал.
        commandService.sendCommand(conferenceDto, peerId, uid, command);
    }

    private void sendCommandToAll(String uri, PassportOrYaTeamUid uid, String excludePeerId, Command command)
    {
        ConferenceDto conferenceDto = conferenceService.getConferenceByFullUri(uri);
        commandService.sendCommandToAll(conferenceDto, uid, command, excludePeerId);
    }
}
