package ru.yandex.chemodan.app.uaas.client;

import java.util.Arrays;

import javax.annotation.Nonnull;

import org.apache.http.Header;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.uaas.parser.UserAgentObject;
import ru.yandex.chemodan.cache.TimeLimitedCacheFast;
import ru.yandex.chemodan.http.UaasHeadersHolder;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class UaasClient {
    private static final Logger logger = LoggerFactory.getLogger(UaasClient.class);
    private static final ListF<String> UNKNOWN_DEVICE_ID = Cf.x(new String[]{"unknown", "null", "nil", "none", ""});

    private HttpClient httpClient;
    private String host;

    private final TimeLimitedCacheFast<UaasCacheKey, ListF<String>> rawExpHeadersCache;
    private final TimeLimitedCacheFast<UaasCacheKey, ListF<String>> longRawExpHeadersCache;

    public UaasClient(HttpClient httpClient, String host, int cacheTtlMinutes, int longCacheTtlMinutes) {
        this.httpClient = httpClient;
        this.host = host;

        this.rawExpHeadersCache = buildCache("uaas-experiments", cacheTtlMinutes);
        this.longRawExpHeadersCache = buildCache("long-uaas-experiments", longCacheTtlMinutes);
    }

    private TimeLimitedCacheFast<UaasCacheKey, ListF<String>> buildCache(
            String s, int cacheTtlMinutes)
    {
        final TimeLimitedCacheFast<UaasCacheKey, ListF<String>> cache =
                TimeLimitedCacheFast.<UaasCacheKey, ListF<String>>builder()
                        .cacheName(s)
                        .initialCapacity(2 ^ 16)
                        .loadFactor(0.6f)
                        .ttl(() -> cacheTtlMinutes)
                        .build();
        logger.info("Cache enabled with ttl: {}", cacheTtlMinutes);
        return cache;
    }

    public ListF<String> getRawExpHeaders(Option<Long> uid, Option<UserAgentObject> userAgentObject)
    {
        ListF<Header> headers = UaasHeadersHolder.get();
        return rawExpHeadersCache.getFromCacheSome(
                new UaasCacheKey(uid, userAgentObject, headers), this::getRawExpHeadersImpl);
    }

    private ListF<String> getRawExpHeadersImpl(UaasCacheKey uaasCacheKey)
    {
        final Option<Long> uid = uaasCacheKey.getUid();
        final Option<UserAgentObject> userAgentObject = uaasCacheKey.getUserAgentObject();
        final Option<String> cookie = constructCookie(userAgentObject);
        final Option<String> fakeUserAgent = constructFakeUserAgent(userAgentObject);
        final ListF<Header> headers = uaasCacheKey.getHeaders();
        ListF<String> response;
        try {
            response = ApacheHttpClientUtils.execute(buildUaasRequest(uid, cookie, fakeUserAgent, headers), httpClient,
                            (resp) -> Cf.x(resp.getAllHeaders())
                                    .filter(header -> "X-Yandex-ExpFlags".equals(header.getName()))
                                    .map(Header::getValue))
                                    .flatMap(s -> Cf.x(s.split(",")));
            longRawExpHeadersCache.putInCache(uaasCacheKey, response);
        } catch (RuntimeException ex) {
            logger.error("Could not get response from UAAS. Fetching the result from cache", ex);
            response = longRawExpHeadersCache.getFromCache(
                    new UaasCacheKey(uid, userAgentObject, headers)).orElseThrow(() -> ex);
        }
        return response;
    }

    private Option<String> constructFakeUserAgent(Option<UserAgentObject> userAgentObjectO) {
        return userAgentObjectO.map(userAgentObject -> {
                if (StringUtils.startsWithIgnoreCase(userAgentObject.os, "ios")) {
                    if (userAgentObject.device.isMatch(s -> s.equalsIgnoreCase("phone"))) {
                        return "Mozilla/5.0 (iPhone; CPU iPhone OS 11_1 like Mac OS X) AppleWebKit/604.3.5 (KHTML, like Gecko) Version/11.0 Mobile/15B93 Safari/604.1";
                    }
                    if (userAgentObject.device.isMatch(s -> s.equalsIgnoreCase("tablet"))) {
                        return "Mozilla/5.0 (iPad; CPU OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13B143 Safari/601.1";
                    }
                }
                if (StringUtils.startsWithIgnoreCase(userAgentObject.os, "android")) {
                    if (userAgentObject.device.isMatch(s -> s.equalsIgnoreCase("phone"))) {
                        return "Mozilla/5.0 (Linux; Android 5.1.1; Nexus 6 Build/LYZ28E) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/64.0.3265.0 Mobile Safari/537.36";
                    }
                    if (userAgentObject.device.isMatch(s -> s.equalsIgnoreCase("tablet"))) {
                        return "Mozilla/5.0 (Linux; Android 4.4; Nexus 5 Build/_BuildID_) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/30.0.0.0 Mobile Safari/537.36";
                    }
                }
                if ("mac".equalsIgnoreCase(userAgentObject.os) || "windows".equalsIgnoreCase(userAgentObject.os)) {
                    return "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/70.0.3538.102 YaBrowser/18.11.1.80";
                }
                return "";
            });
    }

    private Option<String> constructCookie(Option<UserAgentObject> userAgentObject) {
        return userAgentObject.map(UserAgentObject::getId)
                .filter(id -> !UNKNOWN_DEVICE_ID.containsTs(id.toLowerCase()))
                .map(id -> Md5.A.digest(id).toString())
                .map(md5 -> "fuid01=" + md5.substring(0,16));
    }

    @Nonnull
    private HttpGet buildUaasRequest(Option<Long> uidO, Option<String> cookieO, Option<String> fakeUserAgentO,
                                     ListF<Header> headers) {
        UriBuilder builder = UriBuilder.cons(host).appendPath("disk");
        uidO.map(uid -> builder.addParam("uuid", uid));

        final HttpGet httpGet = new HttpGet(builder.toUrl());
        cookieO.ifPresent(cookie -> httpGet.addHeader("Cookie", cookie));
        fakeUserAgentO.filterNot(String::isEmpty).ifPresent(fakeUserAgent -> httpGet.addHeader("User-Agent", fakeUserAgent));
        headers.forEach(httpGet::addHeader);
        logger.debug("HTTP request to UAAS {} with headers: {}", httpGet, Arrays.toString(httpGet.getAllHeaders()));
        return httpGet;
    }
}
