package ru.yandex.chemodan.app.uaas.experiments;

import java.io.IOException;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.uaas.client.UaasClient;
import ru.yandex.chemodan.app.uaas.flags.ExperimentHeaderParser;
import ru.yandex.chemodan.app.uaas.parser.UaasConditionParser;
import ru.yandex.chemodan.app.uaas.parser.UserAgentObject;
import ru.yandex.chemodan.app.uaas.zk.ExperimentOverride;
import ru.yandex.chemodan.app.uaas.zk.UaasOverrideController;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class ExperimentsManager {
    private static final Logger logger = LoggerFactory.getLogger(ExperimentsManager.class);
    private static final ObjectMapper objectMapper = new ObjectMapper();

    private final UaasConditionParser uaasConditionParser;
    private final UaasOverrideController uaasOverrideController;
    private final UaasClient uaasClient;

    public ExperimentsManager(UaasConditionParser uaasConditionParser, UaasOverrideController uaasOverrideController, UaasClient uaasClient) {

        this.uaasConditionParser = uaasConditionParser;
        this.uaasOverrideController = uaasOverrideController;
        this.uaasClient = uaasClient;
    }

    public JsonNode getJson(Option<Long> uid, Option<UserAgentObject> userAgentObject) {
        final ListF<String> experimentsList = uaasClient.getRawExpHeaders(uid, userAgentObject);
        final ListF<JsonNode> filtered = filterExperiments(experimentsList, uid, userAgentObject);
        return constructJson(filtered);
    }

    private JsonNode constructJson(ListF<JsonNode> filtered) {
        final ObjectNode result = objectMapper.createObjectNode();
        result.withArray("uas_exp_flags").addAll(filtered);
        return result;
    }

    private ListF<JsonNode> filterExperiments(ListF<String> experimentsList, Option<Long> uid, Option<UserAgentObject> userAgentObject) {
        final ListF<JsonNode> flatExperimentsList = experimentsList
                .flatMap(ExperimentHeaderParser::parseRawExpHeaderValuesToJson)
                .map(arrayNode -> StreamSupport.stream(arrayNode.spliterator(), false))
                .flatMap(stream -> stream.collect(Collectors.toList()));
        logger.trace("Experiments for uid {} and user agent {} from UAAS: {}", uid, userAgentObject, flatExperimentsList);
        final ListF<JsonNode> experimentsListWithOverrides =
                getExperimentsListWithOverrides(uid, userAgentObject, flatExperimentsList);
        logger.trace("Experiments with overrides: {}", experimentsListWithOverrides);
        final ListF<JsonNode> filteredByCondition = filterByCondition(userAgentObject, experimentsListWithOverrides);
        logger.trace("Experiments filtered by condition: {}", filteredByCondition);
        return filteredByCondition;
    }

    private ListF<JsonNode> getExperimentsListWithOverrides(Option<Long> uid,
            Option<UserAgentObject> userAgentObject, ListF<JsonNode> flatExperimentsList)
    {
        final Option<String> deviceId = userAgentObject.filterMap(UserAgentObject::getDevice);
        ListF<ExperimentOverride> experimentOverrides = uaasOverrideController.find(uid, deviceId);
        final ListF<JsonNode> overrideNodes = experimentOverrides.filterMap(override -> {
            try {
                return Option.of(objectMapper.readTree(override.getOverrideJson()));
            } catch (IOException e) {
                logger.error("failed to parse override json string with id {}: {}", override.getId(), e);
                return Option.empty();
            }
        });
        logger.trace("Override nodes: {}", experimentOverrides);
        final ListF<Integer> testIdsToRemove = experimentOverrides.flatMap(ExperimentOverride::getTestIds);
        logger.trace("Test ids to remove: {}", testIdsToRemove);
        return flatExperimentsList
                .filter(exp -> !containsTestId(exp, testIdsToRemove))
                .plus(overrideNodes);
    }

    private ListF<JsonNode> filterByCondition(Option<UserAgentObject> userAgentObject,
            ListF<JsonNode> experimentsListWithOverrides)
    {
        return experimentsListWithOverrides
//                      .filter(exp -> uaasConditionParser.validate(Option.ofNullable("os!='iOS'"), userAgentObject)); //this is intended for a quick test of conditions
                .filter(exp -> Option.ofNullable(exp.get("HANDLER")).isMatch(handler -> handler.isTextual() && handler.textValue().equals("DISK")))
                .filter(exp -> uaasConditionParser.validate(getConditionNodeValue(exp), userAgentObject));
    }

    private Option<String> getConditionNodeValue(JsonNode exp) {
        return Option.ofNullable(exp.get("CONDITION")).filter(JsonNode::isTextual).map(JsonNode::textValue);
    }

    public ListF<String> getFlags(long uid) {
        final ListF<String> experimentsList = uaasClient.getRawExpHeaders(Option.of(uid), Option.empty());
        final ListF<JsonNode> filtered = filterExperiments(experimentsList, Option.of(uid), Option.empty());
        return getFlagsFromJson(filtered);
    }

    public ListF<String> getFlags(long uid, String deviceId) {
        final UserAgentObject userAgentObject = UserAgentObject.builder().device(Option.of(deviceId)).build();
        final ListF<String> experimentsList = uaasClient.getRawExpHeaders(Option.of(uid), Option.of(userAgentObject));
        final ListF<JsonNode> filtered = filterExperiments(experimentsList, Option.of(uid), Option.of(userAgentObject));
        return getFlagsFromJson(filtered);
    }

    private boolean containsTestId(JsonNode node, ListF<Integer> testIds) {
        return Option.ofNullable(node.get("CONTEXT"))
                .filterMap(context->Option.ofNullable(context.get("DISK")))
                .filterMap(disk->Option.ofNullable(disk.withArray("testid")))
                .map(jsonNode -> Cf.x(jsonNode.elements()).exists(expTestId->testIds.containsTs(expTestId.asInt())))
                .isSome(true);
    }

    private ListF<String> getFlagsFromJson(ListF<JsonNode> nodes) {
        ListF<String> flags = nodes.filterMap(node -> Option.ofNullable(node.get("CONTEXT")))
                .filterMap(context -> Option.ofNullable(context.get("DISK")))
                .filterMap(disk -> Option.ofNullable(disk.withArray("flags")))
                .flatMap(jsonNode -> Cf.x(jsonNode.elements()).toList())
                .filter(JsonNode::isTextual)
                .map(JsonNode::textValue);

        logger.debug("Experiment flags: {}", flags);
        return flags;
    }
}
