package ru.yandex.chemodan.app.uaas.zk;

import java.io.IOException;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class UaasOverrideController {
    private static final Logger logger = LoggerFactory.getLogger(UaasOverrideController.class);

    private static final ObjectMapper objectMapper = new ObjectMapper();

    private final UaasOverrideZkRegistry overrideZkRegistry;
    private final UserGroupsZkRegistry groupsZkRegistry;

    public UaasOverrideController(UaasOverrideZkRegistry overrideZkRegistry, UserGroupsZkRegistry groupsZkRegistry) {
        this.overrideZkRegistry = overrideZkRegistry;
        this.groupsZkRegistry = groupsZkRegistry;
    }

    public ExperimentOverride get(String id){
        return overrideZkRegistry.get(id);
    }

    public Option<ExperimentOverride> getO(String id) {
        return overrideZkRegistry.getO(id);
    }

    public CollectionF<ExperimentOverride> getAll(){
        return overrideZkRegistry.getAll();
    }

    public void put(String id, ListF<String> uids, ListF<Integer> testIds, Option<String> description, String overrideJsonString){
        try {
            JsonNode node = objectMapper.readTree(overrideJsonString);

            Validate.isTrue(node.isObject(), "override json must be object");
            Validate.isTrue("DISK".equals(node.get("HANDLER").textValue()), "override json must have HANDLER: \"DISK\" on top level");
            Validate.isTrue(node.get("CONTEXT") != null, "override json must have CONTEXT on top level");
            Validate.isTrue(node.get("CONTEXT").isObject(), "CONTEXT must be an object");

            Validate.isTrue(node.get("CONTEXT").get("DISK") != null, "override json must have CONTEXT.DISK on top level");
            Validate.isTrue(node.get("CONTEXT").get("DISK").isObject(), "CONTEXT.DISK must be an object");

            JsonNode internalNode = node.get("CONTEXT").get("DISK");
            checkNodeIsStringArray("flags", internalNode.get("flags"));
            checkNodeIsStringArray("testid", internalNode.get("testid"));
        } catch (IOException e) {
            logger.error("failed to parse override json string with id {}: {}", id, e);
            return;
        }

        ExperimentOverride override = ExperimentOverride.builder()
                .id(id)
                .uids(uids)
                .testIds(testIds)
                .overrideJson(overrideJsonString)
                .description(description)
                .build();
        overrideZkRegistry.put(override);
    }

    private void checkNodeIsStringArray(String name, JsonNode node) {
        Validate.isTrue(node != null, "no fiel " + name);
        Validate.isTrue(node.isArray(), name + " must be an array");

        ArrayNode array = (ArrayNode) node;
        for (JsonNode item : Cf.x(array.elements()).toList()) {
            Validate.isTrue(item.isTextual(), name + " array contains non-string value");
        }
    }

    public void remove(String id){
        overrideZkRegistry.remove(id);
    }

    public ListF<ExperimentOverride> find(Option<Long> uid, Option<String> deviceId) {
        ListF<String> groupIds = groupsZkRegistry.getAll().filter(group -> {
            if (uid.isPresent() && group.uids.containsTs(uid.get().toString())) {
                return true;
            }

            return deviceId.isPresent() && group.uids.containsTs("device_id:" + deviceId.get());
        }).map(UserGroupsZkRegistry.UserGroup::getId);

        final CollectionF<ExperimentOverride> filtered = getAll()
                .filter(e -> e.getUids().containsTs(uid.map(String::valueOf).getOrNull())
                        || e.getUids().map(expDeviceId -> StringUtils.substringAfter(expDeviceId, "device_id:")).containsTs(deviceId.getOrNull())
                        || e.getUids().exists(groupIds::containsTs)
                );
        return Cf.toList(filtered);
    }
}
