package ru.yandex.chemodan.uploader.av.drweb;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.av.BaseAntivirus;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.misc.time.TimeUtils;

/**
 * TODO drwebc can accept file with offset/length, it might be faster
 * TODO limit number of concurrent checks (CPU-bound; drwebd preforks several processes)
 * TODO propertize drwebdc invocation
 *
 * @author vavinov
 * @author akirakozov
 */
public class ShellAntivirus extends BaseAntivirus {
    private static final Logger logger = LoggerFactory.getLogger(ShellAntivirus.class);

    private static final Duration DEFAULT_TIMEOUT = Duration.standardMinutes(5);

    private final String command;
    private final boolean useOnlyStdin;

    public ShellAntivirus(String command, boolean useOnlyStdin, boolean enabled,
                          DataSize maxFileSize, boolean supportFileOnly)
    {
        super(enabled, maxFileSize, supportFileOnly);
        this.command = command;
        this.useOnlyStdin = useOnlyStdin;
    }

    protected AntivirusResult doCheck(InputStreamSource file) {
        boolean processAsFile = (file instanceof File2) && !useOnlyStdin;
        String[] commands = Cf.list(command.split(" "))
                .plus1(processAsFile ? ((File2) file).getAbsolutePath() : "-").toArray(String.class);

        Duration timeout = ThreadLocalTimeout.timeout().getOrElse(DEFAULT_TIMEOUT);

        logger.debug("Antivirus command line: " + StringUtils.join(commands, " "));
        ProcessBuilder pb = new ProcessBuilder(commands);
        Process process = null;
        try {
            process = pb.start();
            if (!processAsFile) {
                BufferedOutputStream out = new BufferedOutputStream(process.getOutputStream());
                file.readTo(out);
                out.close();
            }

            try {
                boolean result = process.waitFor(timeout.getMillis(), TimeUnit.MILLISECONDS);
                if (!result) {
                    String after = "after " + TimeUtils.millisecondsToSecondsString(timeout.getMillis());
                    throw new ThreadLocalTimeoutException("timed out " + after);
                }

                int code = process.exitValue();

                String stdout = InputStreamSourceUtils.wrap(process.getInputStream()).readText();
                logger.info("Antivirus processed status code: {}, stdout: [{}]", code, stdout);

                if (code == 0) {
                    return AntivirusResult.HEALTHY;
                } else if ((code & 1) == 1) {
                    return AntivirusResult.INFECTED;
                } else {
                    return AntivirusResult.UNKNOWN;
                }
            } catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw ExceptionUtils.translate(ex);
            }
        } catch (IOException ex) {
            throw IoUtils.translate(ex);
        } finally {
            if (process != null) process.destroy();
        }
    }
}
