package ru.yandex.chemodan.uploader.av.icap;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * ICAP imasdfasFC: https://tools.ietf.org/html/rfc3507
 * Yandex-Mail implementation: https://github.yandex-team.ru/mail/nwsmtp/blob/master/src/avir_client.cpp#L372
 * One of java client implementation: https://github.com/Baekalfen/ICAP-avscan
 *
 * @author akirakozov
 */
public class IcapClient implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(IcapClient.class);

    private static final Charset UTF8 = StandardCharsets.UTF_8;
    private static final byte[] REQUEST_TERMINATOR = "\r\n0\r\n\r\n".getBytes(UTF8);

    private static final int CHUNK_SIZE = 65536;

    private Socket socket;
    private OutputStream socketOut;
    private InputStream socketIn;

    private final String icapService;
    private final String host;
    private final int port;
    private final IcapClientConfiguration clientConfiguration;


    public IcapClient(String host, int port, String icapService,
            IcapClientConfiguration clientConfiguration) {
        this.host = host;
        this.port = port;
        this.icapService = icapService;
        this.clientConfiguration = clientConfiguration;

        try {
            socket = new Socket();
            int connectionTimeout = this.clientConfiguration.getConnectTimeout();
            try {
                socket.connect(new InetSocketAddress(host, port), connectionTimeout);
            } catch (SocketTimeoutException e) {
                logger.debug("The connection timeout has been exceeded for ICAP client connectionTimeout = {} ms", connectionTimeout);
                throw new IOException("The connection timeout has been exceeded for ICAP client");
            }
            socket.setSoTimeout(this.clientConfiguration.getReadTimeout());

            socketOut = socket.getOutputStream();
            socketIn = socket.getInputStream();
        } catch (IOException e) {
            throw IoUtils.translate(e);
        }
    }

    protected IcapClient() {
        this.icapService = "";
        this.host = "";
        this.port = 0;
        this.clientConfiguration = new IcapClientConfiguration(0, 0, 0, CHUNK_SIZE);
    }

    public AntivirusResult scanFile(InputStreamSource source) {
        if (!source.lengthO().isPresent()) {
            logger.info("Couldn't check input source on viruses, because length is unknown");
            return AntivirusResult.UNKNOWN;
        }
        InputStream input = source.getInputUnchecked();
        try {
            writeRequest(input, source.lengthO().get());
            return getAntivirusResult();
        } catch (IOException e) {
            throw IoUtils.translate(e);
        } finally {
            IoUtils.closeQuietly(input);
        }
    }

    @Override
    public void close() throws IOException {
        socket.close();
    }

    protected AntivirusResult getAntivirusResult() {
        return new IcapResponseStreamParser(getIcapResponseProvider()).getAntivirusResult();
    }

    private void writeRequest(InputStream input, long size) throws IOException {
        long writeStart = System.currentTimeMillis();
        socketOut.write(prepareRequestHeaders(size));
        int nread;
        byte[] buf = new byte[CHUNK_SIZE];
        while ((nread = input.read(buf)) > 0) {
            validateWritingTimeouts(writeStart);
            socketOut.write(buf, 0, nread);
        }
        socketOut.write(REQUEST_TERMINATOR);
    }

    private void validateWritingTimeouts(long startWritingTime) throws IOException {
        int writingTime = (int) (System.currentTimeMillis() - startWritingTime);
        if (clientConfiguration.getWriteTimeout() <= writingTime) {
            throw new IOException(String.format("ICAP Client took too long time to write request (%s ms)", writingTime));
        }
    }

    private byte[] prepareRequestHeaders(long dataSize) {
        return ("RESPMOD icap://" + host + ":" + port + "/" +icapService +" ICAP/1.0\r\n"
                + "Host: " + host + "\r\n"
                + "Allow: 204\r\n"
                + "Encapsulated: res-body=0\r\n"
                + "Connection: close\r\n"
                + "\r\n"
                + Long.toHexString(dataSize) + "\r\n").getBytes(UTF8);
    }

    protected IcapResponseProvider getIcapResponseProvider() {
        return new IcapResponseInputStreamProvider(socketIn, clientConfiguration.getReadChunkSize());
    }

}
