package ru.yandex.chemodan.uploader.av.icap;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.misc.lang.StringUtils;

public class IcapResponseStreamParser {

    private static final String CRLF = "\r\n";
    private static final String TERMINATOR = "\r\n\r\n";
    private static final String HTTP_STATUS_LINE_PREFIX = "HTTP/";

    private final IcapResponseProvider icapResponseProvider;

    public IcapResponseStreamParser(IcapResponseProvider icapResponseProvider) {
        this.icapResponseProvider = icapResponseProvider;
    }

    public AntivirusResult getAntivirusResult() {
        StringBuilder response = new StringBuilder();
        int icapStatusCode = getIcapStatusCode(response);
        if (icapStatusCode != 200 && icapStatusCode != 204) {
            throw new IcapException("Icap error code: " + icapStatusCode);
        }
        Option<Integer> httpStatusCode = getHttpStatusCode(response);
        if (!httpStatusCode.isSome(403)) {
            return AntivirusResult.HEALTHY;
        }
        return getResult(response);
    }

    private int getIcapStatusCode(StringBuilder response) {
        while (!icapResponseProvider.endHasBeenReached() && response.indexOf(CRLF) == -1) {
            response.append(icapResponseProvider.nextResponsePart());
        }
        try {
            String statusLine = StringUtils.substringBefore(response.toString(), CRLF);
            return Integer.parseInt(statusLine.split(" ")[1]);
        } catch (Exception e) {
            throw new IcapException("Couldn't parse icap status code: " + response, e);
        }
    }

    private Option<Integer> getHttpStatusCode(StringBuilder response) {
        while (!icapResponseProvider.endHasBeenReached() && response.indexOf(TERMINATOR) == -1) {
            response.append(icapResponseProvider.nextResponsePart());
        }
        String httpResponsePart = StringUtils.substringAfter(response.toString(), TERMINATOR);
        while (!icapResponseProvider.endHasBeenReached() && httpResponsePart.indexOf(CRLF) == -1) {
            response.append(icapResponseProvider.nextResponsePart());
            httpResponsePart = StringUtils.substringAfter(response.toString(), TERMINATOR);
        }
        try {
            String httpStatusLine = StringUtils.substringBefore(httpResponsePart, CRLF);
            if (StringUtils.startsWith(httpStatusLine, HTTP_STATUS_LINE_PREFIX)) {
                return Option.of(Integer.parseInt(httpStatusLine.split(" ")[1]));
            } else {
                return Option.empty();
            }
        } catch (Exception e) {
            throw new IcapException("Couldn't parse icap status code: " + response, e);
        }
    }

    private AntivirusResult getResult(StringBuilder response) {
        try {
            StringBuilder httpResponse = new StringBuilder(StringUtils.substringAfter(response.toString(), TERMINATOR));
            while (!icapResponseProvider.endHasBeenReached() && httpResponse.indexOf(TERMINATOR) == -1) {
                String responsePart = icapResponseProvider.nextResponsePart();
                response.append(responsePart);
                httpResponse.append(responsePart);
            }
            String httpBody = StringUtils.substringAfter(httpResponse.toString(), TERMINATOR);
            String[] parts = httpBody.split(CRLF);
            while (!icapResponseProvider.endHasBeenReached() && parts.length < 2) {
                String responsePart = icapResponseProvider.nextResponsePart();
                response.append(responsePart);
                httpResponse.append(responsePart);
                httpBody = StringUtils.substringAfter(httpResponse.toString(), TERMINATOR);
                parts = httpBody.split(CRLF);
            }
            if (parts.length >= 2) {
                return "infected".equals(parts[1]) ? AntivirusResult.INFECTED : AntivirusResult.UNKNOWN;
            }
            return AntivirusResult.UNKNOWN;
        } catch (Exception e) {
            throw new IcapException("Couldn't parse encapsulated http body: " + response, e);
        }
    }
}
