package ru.yandex.chemodan.uploader.config;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.uploader.UploaderPorts;
import ru.yandex.chemodan.uploader.http.UploaderDataJetty;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.web.ApiPrivateUrls;
import ru.yandex.chemodan.uploader.web.data.CrossdomainXmlServlet;
import ru.yandex.chemodan.util.ping.CoolPingServlet;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.WtdFilter;
import ru.yandex.misc.web.servletContainer.SingleWarJetty;
import ru.yandex.misc.web.servletContainer.jetty.MuteErrorHandler;

/**
 * @author vavinov
 */
@Configuration
@Import(UploaderDataWorkerContextConfiguration.class)
public class UploaderDataDaemonContextConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(UploaderDataDaemonContextConfiguration.class);

    @Bean
    @Qualifier("dataJetty")
    public SingleWarJetty dataJetty(
            @Value("${http.maxThreads}")
            int maxThreads,
            @Value("${http.maxQueueLength}")
            int maxQueueLength,
            @Value("${ssl.keystore.file}")
            String keystoreFile,
            @Value("${ssl.keystore.password}")
            String keystorePass,
            @Value("${ssl.ciphers}")
            String ciphers,
            @Value("${data.socket.maxIdleTime}")
            Duration maxIdleTime,
            UploaderPorts uploaderHttpPorts,
            UploaderCoreContextConfiguration core,
            UploaderDataWorkerContextConfiguration worker,
            UploadRegistry<MpfsRequestRecord> uploaderRegistry,
            CrossdomainXmlServlet crossdomainXmlServlet,
            CoolPingServlet coolPingServlet)
    {
        SingleWarJetty jetty = new UploaderDataJetty(uploaderRegistry);
        jetty.setHttpPort(uploaderHttpPorts.getDataPort());
        if (!keystoreFile.isEmpty()) {
            jetty.setHttpsPort(uploaderHttpPorts.getDataHttpsPort());
            jetty.setHttpsSslProtocol("https");
            jetty.setHttpsKeystoreFile(keystoreFile);
            jetty.setHttpsKeystorePass(keystorePass);
            jetty.setExcludedSslProtocols(Cf.list("SSLv3"));
            ListF<String> cipherSuites = Cf.list(StringUtils.split(ciphers, ","));
            if (cipherSuites.isNotEmpty()) {
                jetty.setIncludedCipherSuites(cipherSuites);
            }
            jetty.setUseStrongCipherSuites(true);
        } else {
            logger.warn("Keystore file unavailable, disabling SSL.");
        }
        jetty.setMaxThreads(maxThreads);
        jetty.setMaxQueueLength(maxQueueLength);
        jetty.setCopyBeanPostprocessors(true);
        jetty.setLookupServletsInContext(false);
        jetty.setRequestLogFactory(DiskLog4jRequestLog.consF());
        jetty.setMaxIdleTimeMillis((int) maxIdleTime.getMillis());

        jetty.addFilterMapping("/*", new WtdFilter());
        jetty.addServletMapping("/ping", coolPingServlet);
        jetty.addServletMapping(ApiPrivateUrls.UPLOAD_TARGET + "/*", worker.uploadTargetServlet());
        jetty.addServletMapping(ApiPrivateUrls.PATCH_TARGET + "/*", worker.uploadDeltaTargetServlet());
        jetty.addServletMapping(ApiPrivateUrls.UPLOAD_TO_SERVICE_TARGET + "/*", worker.uploadToServiceTargetServlet());
        jetty.addServletMapping(ApiPrivateUrls.SYNC_UPLOAD_TARGET + "/*", worker.syncUploadServlet());
        jetty.addServletMapping(ApiPrivateUrls.UPLOAD_TARGET_FROM_OFFICE + "/*", worker.uploadFromOfficeServlet());
        jetty.addServletMapping("/crossdomain.xml", crossdomainXmlServlet);
        jetty.addServletMapping("/robots.txt", core.robotsTxtServlet());
        jetty.initialize();
        jetty.getContainerRootContext().setErrorHandler(new MuteErrorHandler());
        return jetty;
    }
}
