package ru.yandex.chemodan.uploader.docviewer;

import java.net.URI;

import javax.annotation.PreDestroy;

import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.dom4j.Element;
import org.springframework.util.CollectionUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.preview.PreviewImageManager;
import ru.yandex.chemodan.util.http.ContentTypeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUidOrZero;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4InputStreamSource;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.apache.v4.ReadRootDom4jElementResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ReadStringResponseHandler;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.mime.detect.MimeTypeDetector;

/**
 * http://wiki.yandex-team.ru/DocViewer/Actions
 *
 * @author akirakozov
 * @author qwwdfsad
 * @author vavinov
 */
public class DocviewerClientImpl implements DocviewerClient {
    private static final Logger logger = LoggerFactory.getLogger(DocviewerClientImpl.class);
    private static final BenderJsonParser<ConvertationInfo> parser = Bender.jsonParser(ConvertationInfo.class);

    private final String docviewerWorkersClusterUrl;
    private final String docviewerBackendClusterUrl;
    private final DataSize docviewerMaxFileLength;
    private volatile boolean useRemoteDvInfo;

    private final HttpClient httpClient;

    private volatile ConvertationInfo convertationInfo;

    private final DynamicProperty<ListF<String>> documentMimeTypes =
            new DynamicProperty<>("dv.client.document-mime-types",
                    Cf.list(
                            "application/word",
                            "application/msexcel",
                            "application/powerpoint",
                            "application/vnd.openxmlformats-officedocument.wordprocessingml.template",
                            "text/pdf",
                            "x-application/vnd.ms-spreadsheetml",
                            "image/x-djvu",
                            "application/mspowerpoint",
                            "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
                            "application/mspowerpnt",
                            "application/x-shellscript",
                            "application/vnd.ms-excel",
                            "text/rtf",
                            "application/x-msword",
                            "application/pdf",
                            "application/vnd.ms-word",
                            "application/msword",
                            "application/winword",
                            "application/x-powerpoint",
                            "application/epub+zip",
                            "text/x-pdf",
                            "application/rtf",
                            "application/vnd.oasis.opendocument.text",
                            "application/ms-powerpoint",
                            "application/vnd.openxmlformats-officedocument.presentationml.presentation",
                            "application/x-msexcel",
                            "text/csv",
                            "application/acrobat",
                            "application/vnd.oasis.opendocument.presentation",
                            "application/x-pdf",
                            "x-application/vnd.ms-wordml",
                            "applications/vnd.pdf",
                            "application/djvu",
                            "image/vnd.djvu",
                            "application/x-msw6",
                            "application/x-ms-excel",
                            "text/comma-separated-values",
                            "application/xls",
                            "application/vnd.oasis.opendocument.spreadsheet",
                            "text/plain, application/vnd-mspowerpoint",
                            "application/x-excel",
                            "application/x-dos_ms_excel",
                            "application/x-fictionbook+xml",
                            "application/vnd.msword",
                            "application/vnd.openxmlformats-officedocument.spreadsheetml.template",
                            "application/vnd.ms-powerpoint",
                            "application/doc",
                            "application/vnd.openxmlformats-officedocument.presentationml.template",
                            "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
                    ));


    public DocviewerClientImpl(
            String docviewerWorkersClusterUrl, String docviewerBackendClusterUrl,
            DataSize docviewerMaxFileLength, boolean useRemoteDvInfo, HttpClient httpClient)
    {
        this.docviewerWorkersClusterUrl = docviewerWorkersClusterUrl;
        this.docviewerBackendClusterUrl = docviewerBackendClusterUrl;
        this.docviewerMaxFileLength = docviewerMaxFileLength;
        this.useRemoteDvInfo = useRemoteDvInfo;
        this.httpClient = httpClient;
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    @Override
    public InputStreamSource getDocumentPreview(MulcaId mulcaId, PassportUidOrZero uid, String contentType) {
        HttpGet get = new HttpGet(UrlUtils.addParameters(docviewerWorkersClusterUrl + "/preview",
                Cf.<String, Object>map("uid", uid.getUid())
                        .plus1("url", mulcaId.asMulcaUri())
                        .plus1("unsafe", true)
                        .plus1("content-type", contentType)));

        return new ApacheHttpClient4InputStreamSource(httpClient, addAuthHeader(get));
    }

    @Override
    public InputStreamSource getSourceInternal(PassportUidOrZero uid, String docviewerDownloadUUID)
    {
        MapF<String, Object> paramsTemplate = Cf.<String, Object>map("uid", uid.getUid())
                .plus1("uid", uid.getUid())
                .plus1("download-file-uuid", docviewerDownloadUUID);

        HttpGet get =
                new HttpGet(UrlUtils.addParameters(docviewerBackendClusterUrl + "/source-internal", paramsTemplate));
        get.setHeader("X-Req-Id", String.valueOf(uid.hashCode()));
        return new ApacheHttpClient4InputStreamSource(httpClient, addAuthHeader(get));
    }


    @Override
    public void startConvertToHtml(PassportUidOrZero uid, String sourceUrl, Option<String> contentType) {
        startConvertToHtml(uid, sourceUrl, contentType, false);
    }

    @Override
    public void startConvertToHtml(PassportUidOrZero uid, String sourceUrl, Option<String> contentType,
            boolean warmup)
    {
        String url = UrlUtils.addParameters(docviewerWorkersClusterUrl + "/start",
                Cf.<String, Object>map("uid", uid.getUid())
                        .plus1("url", sourceUrl)
                        .plus1("type", "HTML_WITH_IMAGES"));

        if (contentType.isPresent()) {
            url = UrlUtils.addParameter(url, "content-type", contentType);
        }

        if (warmup) {
            url = UrlUtils.addParameter(url, "warm-up", true);
        }

        execute(new HttpGet(url), new ReadStringResponseHandler());
    }

    private <T> T execute(HttpGet get, ResponseHandler<T> handler) {
        return ApacheHttpClientUtils.execute(addAuthHeader(get), httpClient, handler);
    }

    @Override
    public Element urlInfo(PassportUidOrZero uid, String sourceUrl) {
        String url = UrlUtils.addParameters(docviewerWorkersClusterUrl + "/urlinfo",
                Cf.<String, Object>map("uid", uid.getUid(), "url", sourceUrl));

        return execute(new HttpGet(url), new ReadRootDom4jElementResponseHandler());
    }

    @Override
    public InputStreamSource convertToMsOoxmlFormat(URI url, PassportUidOrZero uid,
            Option<String> archivePath, Option<String> contentType)
    {
        MapF<String, Object> parameters = Cf.<String, Object>map("uid", uid.getUid())
                .plus1("url", url)
                .plus1("unsafe", url.toString().startsWith("mulca://"));
        if (archivePath.isPresent()) {
            parameters = parameters.plus1("archive-path", archivePath.get());
        }
        if (contentType.isPresent()) {
            parameters = parameters.plus1("content-type", contentType.get());
        }
        return new ApacheHttpClient4InputStreamSource(httpClient, UrlUtils.uri(
                UrlUtils.addParameters(docviewerWorkersClusterUrl + "/convert-to-ms-ooxml", parameters)));
    }

    private HttpGet addAuthHeader(HttpGet get) {
        return get;
    }

    boolean isSupportedMimeType(String contentType) {
        if (remoteEnriched()) {
            return convertationInfo.previewMimeTypes.containsTs(contentType) && !isFiltered(contentType);
        } else {
            return documentMimeTypes.get().find(s -> contentType.startsWith(s)).isPresent();
        }
    }

    private boolean isFiltered(String mimeType) {
        return mimeType.startsWith("image/")
                && !"image/x-djvu".equals(mimeType)
                && !"image/vnd.djvu".equals(mimeType);
    }

    @Override
    public boolean isDocumentSupportedByDocviewer(Option<String> contentType, Option<Long> sourceLength) {
        boolean knownContentType = contentType.isPresent();
        boolean validSize = sourceLength.isPresent()
                && sourceLength.get() < maxFileLength()
                && sourceLength.get() > 0L;
        boolean canHandle = knownContentType && isSupportedMimeType(contentType.get());

        if (knownContentType && validSize && canHandle) {
            return true;
        } else {
            logger.warn("Will not submit to docviewer: knownContentType={}, validSize={}, canHandle={}",
                    knownContentType, validSize, canHandle);
            return false;
        }
    }

    @Override
    public boolean isPreviewSupported(String contentType) {
        return isSupportedMimeType(contentType)
                || ContentTypeUtils.isVideoContentType(Option.of(contentType))
                || PreviewImageManager.isSupportedMimeType(contentType);
    }

    @Override
    public String correctMimeTypeByFileNameIfUnknown(String mimeType, Option<String> filename) {
        if (!isPreviewSupported(mimeType)) {
            return MimeTypeDetector.detect(filename, Option.empty()).getOrElse(mimeType);
        }
        return mimeType;
    }

    private byte[] requestConvertationInfo() {
        return new ApacheHttpClient4InputStreamSource(
                httpClient, new HttpGet(UrlUtils.uri(docviewerWorkersClusterUrl + "/preview-supported")))
                .readBytes();
    }

    @Override
    public void updateCachedConvertationInfo() {
        try {
            logger.debug("Requesting convertation info from DV");
            this.convertationInfo = parser.parseJson(requestConvertationInfo());
        } catch (Exception e) {
            logger.error("Uncaught exception in DV info extractor", e);
        }
    }

    ConvertationInfo getConvertationInfo() {
        return convertationInfo;
    }

    void setUseRemoteInfo(boolean useRemote) {
        this.useRemoteDvInfo = useRemote;
    }

    @Override
    public boolean remoteEnabled() {
        return useRemoteDvInfo;
    }

    public boolean remoteEnriched() {
        return remoteEnabled() &&
                convertationInfo != null &&
                convertationInfo.maxFileSize != null &&
                !CollectionUtils.isEmpty(convertationInfo.previewMimeTypes);
    }

    private long maxFileLength() {
        return remoteEnriched()
                ? convertationInfo.maxFileSize.toBytes()
                : docviewerMaxFileLength.toBytes();
    }

}
