package ru.yandex.chemodan.uploader.exif;

import java.io.StringReader;

import org.dom4j.Element;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo.GeoCoords;
import ru.yandex.commune.image.RotateAngle;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.commune.image.imageMagick.PingImageReport;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ShellUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.mime.detect.MimeTypeDetector;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author akirakozov
 */
public class ExifTool {

    private static final Logger logger = LoggerFactory.getLogger(ExifTool.class);

    private static final DateTimeFormatter DATE_TIME_FORMATTER =
            new DateTimeFormatterBuilder()
                .append(ISODateTimeFormat.date())
                .appendLiteral('T')
                .append(ISODateTimeFormat.hourMinuteSecond())
                .toFormatter();

    private static final String EXIF_TOOL_CMD = "/usr/bin/exiftool";
    private static final Duration TIMEOUT = Duration.standardSeconds(10);

    private static final String LONGITUDE_TAG = "GPSLongitude";
    private static final String LONGITUDE_REF_TAG = "GPSLongitudeRef";
    private static final String LATITUDE_TAG = "GPSLatitude";
    private static final String LATITUDE_REF_TAG = "GPSLatitudeRef";
    private static final String DATE_TIME_TAG = "DateTimeOriginal";

    private final String exifToolCommand;

    public static final ExifTool INSTANCE = new ExifTool();

    public ExifTool() {
        this(EXIF_TOOL_CMD);
    }

    public ExifTool(String exifToolCommand) {
        this.exifToolCommand = exifToolCommand;
    }

    private static final SetF<String> supportedMimeTypes = Cf.set(
            "image/jpeg",
            "image/jpg",
            "image/pjpeg",
            "image/png",
            "image/gif",
            "image/tiff",
            "image/bmp",
            "image/vnd.adobe.photoshop",
            "image/psd",
            "image/cr2",
            "image/nef",
            "image/canon raw",
            "image/x-nikon-nef",
            "image/x-canon-cr2",
            "image/x-canon-raw",
            "image/x-png",
            "image/x-bmp",
            "image/arw",
            "image/x-sony-raw",
            "image/x-sony-arw",
            "image/webp",
            "image/heif",
            "image/jp2");

    public static boolean isSupportedMimeType(String mimeType) {
        String normalizedMimeType = MimeTypeDetector.normalizeMimeType(Option.of(mimeType)).getOrElse("");
        return supportedMimeTypes.containsTs(normalizedMimeType);
    }

    public ExifInfo getExif(File2 source, Duration timeout) {
        ListF<String> cmd = Cf.arrayList(exifToolCommand);
        cmd.add("-EXIF:DateTimeOriginal");
        cmd.add("-GPS:all#");

        cmd.add("-m");
        cmd.addAll("-H", "-X", "-q");
        cmd.add(source.getAbsolutePath());

        ExecResult result = runScript(cmd, (int) timeout.getMillis());
        try {
            return parseReport(result.getOutput());
        } catch (Exception e) {
            logger.warn("Unable to parse report: " + result.getOutput());
            throw new RuntimeException("Unable to parse exif report", e);
        }
    }

    public ExifInfo getExif(File2 source) {
        return getExif(source, TIMEOUT);
    }

    public void removeExif(File2 file) {
        ListF<String> cmd = Cf.arrayList(exifToolCommand);

        cmd.add("-m");
        cmd.add("-all=");
        cmd.add(file.getAbsolutePath());

        runScript(cmd);
    }

    public void addExif(File2 source, File2 dest, boolean excludeOrientation) {
        ListF<String> cmd = Cf.arrayList(exifToolCommand);

        cmd.add("-m");
        cmd.add("-q");
        cmd.add("-overwrite_original");
        cmd.add("-tagsFromFile");
        cmd.add(source.getAbsolutePath());
        if (excludeOrientation) {
            cmd.add("--Orientation");
        }
        cmd.add(dest.getAbsolutePath());

        runScript(cmd);
    }

    public String getFullExifInJson(File2 source) {
        ListF<String> cmd = Cf.arrayList(exifToolCommand);

        cmd.add("-m");
        cmd.add("-a");
        cmd.add("-q");
        cmd.add("-j");

        // exclude all tags from group File, ExifTool
        cmd.add("--File:ALL");
        cmd.add("--ExifTool:ALL");

        cmd.add(source.getAbsolutePath());

        ExecResult result = runScript(cmd);
        return result.getStdout();
    }

    public void copyExif(File2 source, File2 dest, boolean excludeOrientation) {
        removeExif(dest);
        addExif(source, dest, excludeOrientation);
    }

    public void addGeoAndCreateTimeTags(
            File2 source, Option<GeoCoords> coordsO, Option<LocalDateTime> creationTimeO)
    {
        if (!coordsO.isPresent() && !creationTimeO.isPresent()) {
            return;
        }

        ListF<String> cmd = Cf.arrayList(exifToolCommand);
        cmd.add("-m");
        cmd.add("-q");

        // add geo coords
        if (coordsO.isPresent()) {
            GeoCoords coords = coordsO.get();
            String latitudeRef = coords.getLatitude()    >= 0. ? "N" : "S";
            String longitudeRef = coords.getLongitude() >= 0. ? "E" : "W";

            cmd.add("-" + LATITUDE_REF_TAG + "=" + latitudeRef);
            cmd.add("-" + LATITUDE_TAG + "=" + coords.getLatitude());
            cmd.add("-" + LONGITUDE_REF_TAG + "=" + longitudeRef);
            cmd.add("-" + LONGITUDE_TAG + "=" + coords.getLongitude());
        }

        if (creationTimeO.isPresent()) {
            cmd.add("-" + DATE_TIME_TAG + "=" + DATE_TIME_FORMATTER.print(creationTimeO.get()));
        }

        cmd.add(source.getAbsolutePath());

        String script = ShellUtils.commandToScript(cmd);
        logger.info("Run exiftool with params: {}", script);
        ExecResult result = ShellUtils.executeGrabbingOutput(script, (int) TIMEOUT.getMillis());
        if (!result.isSuccess()) {
            logger.warn("Can't add geo and time tags");
        }
    }

    private static ExifInfo parseReport(String report) {
        Element e = Dom4jUtils.readRootElement(new StringReader(report)).element("Description");
        Option<Instant> creationDate = Option.empty();
        Option<GeoCoords> geoCoords = Option.empty();

        if (e.element(DATE_TIME_TAG) != null) {
            creationDate = Option.of(ExifXmlParser.parseDataTime(e.elementText(DATE_TIME_TAG)));
        }

        if (e.element(LATITUDE_TAG) != null && e.element(LONGITUDE_TAG) != null) {
            int latitudeSign = 1;
            int longitudeSign = 1;
            if (e.element(LATITUDE_REF_TAG) != null && "S".equalsIgnoreCase(e.elementText(LATITUDE_REF_TAG))) {
                latitudeSign = -1;
            }
            if (e.element(LONGITUDE_REF_TAG) != null && "W".equalsIgnoreCase(e.elementText(LONGITUDE_REF_TAG))) {
                longitudeSign = -1;
            }
            double latitude = ExifXmlParser.calculateCoordinate(e.elementText(LATITUDE_TAG));
            double longitude = ExifXmlParser.calculateCoordinate(e.elementText(LONGITUDE_TAG));
            geoCoords = Option.of(new GeoCoords(latitudeSign * latitude, longitudeSign * longitude));
        }

        return new ExifInfo(creationDate, geoCoords);
    }

    public static int getRotateAngle(File2 image, ImageMagick imageMagick) {
        PingImageReport pingResult = imageMagick.pingImage(image);
        return pingResult.getExifRotateAngle().getOrElse(RotateAngle.D0).getAngle();
    }

    private static ExecResult runScript(ListF<String> cmd, int timeout) {
        String script = ShellUtils.commandToScript(cmd);

        logger.info("Run exiftool with params: {}", script);
        ExecResult result = ShellUtils.executeGrabbingOutput(script, timeout);
        Check.isTrue(result.isSuccess(), "Exiftool exited with error code, err: " + result.getStderr());

        return result;
    }

    private static ExecResult runScript(ListF<String> cmd) {
        return runScript(cmd, (int) TIMEOUT.getMillis());
    }

}
