package ru.yandex.chemodan.uploader.exif;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.math.Fraction;
import org.joda.time.DateTimeConstants;
import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.DateTimeFormatterBuilder;
import org.joda.time.format.DateTimeParser;
import org.joda.time.format.DateTimeParserBucket;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.function.Function1V;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Copy paste from fotki
 * https://github.yandex-team.ru/media/fotki/blob/master/common/src/main/java/ru/yandex/fotki/util/exif/ExifXmlizer.java
 * @author akirakozov
 */
public class ExifXmlParser {
    private static final Logger logger = LoggerFactory.getLogger(ExifXmlParser.class);

    private static final DateTimeFormatter EXIF_DATETIME_PARSER = exifDateTimeParser();
    private static final Pattern EXIF_ZONE_OFFSET_PATTERN = Pattern.compile("^(\\+|\\-)(\\d{1,2}):(\\d{2})");

    public static Instant parseDataTime(String value) {
        try {
            return EXIF_DATETIME_PARSER.parseDateTime(value).toInstant();
        } catch (IllegalArgumentException ex) {
            return ISODateTimeFormat.dateTimeParser().parseDateTime(value).toInstant();
        }
    }

    private static DateTimeFormatter exifDateTimeParser() {
        DateTimeFormatterBuilder builder = new DateTimeFormatterBuilder();
        builder.append(DateTimeFormat.forPattern("yyyy:MM:dd HH:mm:ss"));
        builder.appendOptional(new DateTimeFormatterBuilder().appendLiteral(".").appendMillisOfSecond(2).toParser());
        builder.appendOptional(exifTimeZoneParser());
        return builder.toFormatter().withOffsetParsed();
    }

    private static DateTimeParser exifTimeZoneParser() {
        return new DateTimeParser() {

            @Override
            public int estimateParsedLength() {
                return 6;
            }

            @Override
            public int parseInto(DateTimeParserBucket bucket, String text, int position) {
                if (text.length() == position) {
                    return ~position;
                }
                String str = text.substring(position);
                if (str.charAt(0) == 'Z') {
                    bucket.setOffset(Integer.valueOf(0));
                    return position + 1;
                }

                Matcher matcher = EXIF_ZONE_OFFSET_PATTERN.matcher(str);
                if (matcher.matches()) {
                    position++;
                    int hours = Integer.parseInt(matcher.group(2));
                    if (hours > 23) {
                        return ~position;
                    }
                    position += matcher.end(2);
                    int minutes = Integer.parseInt(matcher.group(3));
                    if (minutes > 59) {
                        return ~position;
                    }
                    int offset = hours * DateTimeConstants.MILLIS_PER_HOUR +
                            minutes * DateTimeConstants.MILLIS_PER_MINUTE;

                    bucket.setOffset(Integer.valueOf(matcher.group(1).equals("-") ? -offset : offset));
                    return position + 2;
                }

                return ~position;
            }
        };
    }

    public static double calculateCoordinate(String value) {
        final double coord[] = new double[] {0};
        forEachCoordinateFraction(value, new Function1V<String>() {
            private double multiplier = 1.0;

            public void apply(String s) {
                s = s.trim();
                if (Character.isLetter(s.charAt(s.length() - 1))) {
                    // Some silly programs put letters like 'E' 'W' 'N' 'S' here.
                    // Let's just ignore it for now since it will be duplicated in other EXIF tags
                    s = s.substring(0, s.length() - 1);
                }
                Fraction fraction = Fraction.getFraction(s);
                double val = (double) fraction.getNumerator() / fraction.getDenominator();
                coord[0] += val * multiplier;
                multiplier /= 60.0;
            }
        });
        return coord[0];
    }

    private static void forEachCoordinateFraction(String fullValue, Function1V<String> f) {
        String[] rawCoords = fullValue.contains(",") ? fullValue.split(",") : fullValue.split(" ");

        try {
            for (String rawCoord : rawCoords) {
                f.apply(rawCoord);
            }
        } catch (Exception e) {
            // do nothing for broken values
            logger.info("Error parsing gps coordinate value {}: {}", fullValue, e);
        }
    }

}
