package ru.yandex.chemodan.uploader.http;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.HandlerWrapper;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.http.JettyHandlerUtil;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * @author akirakozov
 */
public class UploaderDataRequestHandler extends HandlerWrapper {
    private final UploadRegistry<MpfsRequestRecord> uploadRegistry;

    public UploaderDataRequestHandler(UploadRegistry<MpfsRequestRecord> uploadRegistry) {
        this.uploadRegistry = uploadRegistry;
    }

    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request,
                       HttpServletResponse response) throws IOException, ServletException
    {
        RequestIdStack.Handle h = RequestIdStack.push();
        Option<String> ycrid = Option.ofNullable(request.getHeader(CommonHeaders.YANDEX_CLOUD_REQUEST_ID));
        pushNewRequestIdIfNecessary(target, ycrid);
        try {
            super.handle(target, baseRequest, request, response);
        } catch (IOException|ServletException|RuntimeException ex) {
            JettyHandlerUtil.logExceptionAndRemoveStackTrace(request, ex);
            throw ex;
        } finally {
            h.popSafely();
        }
    }

    private void pushNewRequestIdIfNecessary(String target, Option<String> ycrid) {
        // common data handle case, ycrid should be gotten from uploadTarget task
        Option<UploadRequestId> reqId = getUploadRequestId(target);
        if (reqId.isPresent()) {
            String current = RequestIdStack.current().get();
            RequestIdStack.pushReplace(current + getNewRequestId(current, reqId.get()));
            return;
        }

        // upload-target-from-office case
        // in this case, there are no uploadTarget task now, ycrid should be goten from header
        if (ycrid.isPresent()) {
            RequestIdStack.pushReplace(RequestIdStack.current().get() + getYcridPostfix(ycrid.get()));
        }
    }

    public String getNewRequestId(String current, UploadRequestId id) {
        String recordId = (StringUtils.isBlank(current) ? "#" : ",") + id.toString();
        String yandexCloudRequestId = uploadRegistry.findRecordAndDeleteIfEmpty(id)
                .map(r -> r.getRequest().yandexCloudRequestId.getOrElse(""))
                .getOrElse("");

        if (!StringUtils.isEmpty(yandexCloudRequestId)) {
            recordId  += getYcridPostfix(yandexCloudRequestId);
        }
        return recordId;
    }

    private String getYcridPostfix(String ycrid) {
        return ",#ycrid=" + ycrid;
    }

    private Option<UploadRequestId> getUploadRequestId(String target) {
        // path should be like /<handle-name>/<request-id> for all data handlers,
        // except /upload-target-from-office
        String result = StringUtils.substringAfter(target, "/");
        result = StringUtils.substringAfterLast(result, "/");

        return StringUtils.notBlankO(result).map(UploadRequestId::valueOf);
    }

}
