package ru.yandex.chemodan.uploader.log;


import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.registry.record.Xmlizer2;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.uploader.log.DescribableResult;
import ru.yandex.commune.uploader.log.Event;
import ru.yandex.commune.uploader.log.Event.EventType;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.TimeUtils;

/**
 * Events logging.  The logs written by this class will be parsed by admins' monitoring tools (such as awk),
 * so the messages must have a fixed, parse-friendly format.
 *
 * Oleg Leksunin is the ultimate master of this class.  Whatever and however he says is to be logged, must be logged
 * that way.
 *
 * https://jira.yandex-team.ru/browse/CHEMODAN-1683
 *
 * @author vavinov
 */
public class Events {
    private static final Logger logger = LoggerFactory.getLogger(Events.class);
    private static final Logger tskvLogger = LoggerFactory.getLogger("ru.yandex.chemodan.uploader.log.tskv.Events");
    private static final String NONE_VALUE = "none";

    public static void log(StageInfo info, Event event) {
        EventsStatistic.update(info, event);
        log(info, event, false);
        log(info, event, true);
    }

    private static void log(StageInfo info, Event event, boolean isTskvLog) {
        String elementsSeparator = isTskvLog ? "\t" : ", ";

        String successSpecificStr = getSpecificDescriptionPart(event, elementsSeparator);

        if (StringUtils.isNotEmpty(successSpecificStr)) {
            successSpecificStr += elementsSeparator;
        }

        Logger properLogger = isTskvLog ? tskvLogger : logger;
        properLogger.info("type={}" + elementsSeparator +
                        "stage={}" + elementsSeparator +
                        "duration={}" + elementsSeparator +
                        "{}success={}",
                info.taskType, info.stageName,
                TimeUtils.toSecondsString(event.getDuration()),
                successSpecificStr, serializeSuccess(event));
    }

    private static String serializeSuccess(Event event) {
        if (event.getType() == EventType.TEMP_FAILURE) {
            return "unknown";
        } else {
            return Boolean.toString(event.isSuccess());
        }
    }

    private static String getSpecificDescriptionPart(Event event, String elementsSeparator) {
        switch (event.getType()) {
            case SUCCESS:
                return describe(event.getResultO().get()).entries().mkString(elementsSeparator, "=");
            case FAILURE:
            case TEMP_FAILURE:
                return "cause=" + formatCauseForLogging(event.getFailureCause());
            case SKIPPED_FAILURE:
                return "skipped=true" + elementsSeparator + "cause=" + formatCauseForLogging(event.getFailureCause());
            case DISABLED:
                return "disabled=true";
            default:
                throw new IllegalStateException("Unknown event type: " + event.getType());
        }
    }

    static String formatCauseForLogging(String cause) {
        return addQuotes(cause.replace("\n", " ").replace("\t", " "));
    }

    static String addQuotes(String str) {
        return "\"" + str + "\"";
    }

    private static MapF<String, String> describe(Object o) {
        if (o instanceof DescribableResult) {
            return ((DescribableResult) o).describe();
        } else if (o instanceof MulcaId) {
            return Cf.map("mulca_id", ((MulcaId) o).toSerializedString());
        } else if (o instanceof ArchiveListing) {
            ArchiveListing listing = (ArchiveListing) o;
            String listSize = Integer.toString(Xmlizer2.xmlizeArchiveList(listing).length());
            return Cf.map("format", listing.getArchiveFormat(), "result-size", listSize)
                    .plus1("elements", Integer.toString(listing.getEntries().size()));
        } else if (o instanceof Boolean) {
            return Cf.map("result", o.toString());
        } else if (o instanceof Option) {
            if (((Option<?>) o).isPresent()) {
                return describe(((Option<?>) o).get());
            } else {
                return Cf.map("result", NONE_VALUE);
            }
        } else {
            return Cf.map("result", o.toString());
        }
    }
}
