package ru.yandex.chemodan.uploader.log;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.services.ServiceIncomingFile;
import ru.yandex.chemodan.uploader.services.ServiceIncomingImage;
import ru.yandex.commune.uploader.log.Event;
import ru.yandex.commune.uploader.util.UploadSpeedCalculator;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.mime.detect.MediaType;
import ru.yandex.misc.mime.detect.MediaTypeDetector;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.MeterMapWithDistribution;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.monica.util.measure.MeasureInfo;

/**
 * @author Lev Tolmachev
 */
public class EventsStatistic {
    @MonicaMetric(description = "Stages rps, timings and errors rate")
    @GroupByDefault
    private static final InstrumentMap stages = new InstrumentMap();

    @MonicaMetric(description = "Stage results")
    @GroupByDefault
    private static final MeterMap stageResults = new MeterMap();

    @MonicaMetric(description = "Media types")
    @GroupByDefault
    private static final MeterMap mediaTypes = new MeterMapWithDistribution();

    @MonicaMetric(description = "Media types traffic")
    @GroupByDefault
    private static final MeterMap mediaTypesTraffic = new MeterMapWithDistribution();

    @MonicaMetric(description = "Upload time for 1Mb of file")
    @GroupByDefault
    private static final Statistic uploadTime1Mb = new Statistic();

    static {
        MonicaStaticRegistry.register(EventsStatistic.class, new MetricGroupName(
                "events",
                new MetricName("events"),
                "Events statistics. Here can be everithing, that can be found in events-log"));
    }

    public static void update(StageInfo info, Event event) {
        updateStageTime(info, event);
        updateStageResults(info, event);
        updateMediaTypeStatistic(event);
        updateUploadSpeedStatistic(event);
    }

    private static void updateUploadSpeedStatistic(Event event) {
        if (event.getResultO().isPresent() && event.getResultO().get() instanceof IncomingFile) {
            IncomingFile inFile = (IncomingFile) event.getResultO().get();
            double uploadSpeed = UploadSpeedCalculator.computeUploadSpeed(inFile.getUploadedPartsInfo());

            if (uploadSpeed != 0) {
                uploadTime1Mb.update(DataSize.fromMegaBytes(1).toBytes() / uploadSpeed);
            }
        }
    }

    private static void updateMediaTypeStatistic(Event event) {
        if (hasContentInfoResultType(event)) {
            ContentInfo contentInfo = (ContentInfo) event.getResultO().get();
            MediaType mediaType = MediaTypeDetector.detectMediaType(contentInfo.getContentType());
            mediaTypes.inc(mediaType.getName());
            mediaTypesTraffic.inc(contentInfo.getContentLength(), mediaType.getName());
        }
    }

    private static boolean hasContentInfoResultType(Event event) {
        return event.getResultO().isPresent() && event.getResultO().get() instanceof ContentInfo;
    }

    private static void updateStageResults(StageInfo info, Event event) {
        stageResults.inc(event.getType().name().toLowerCase());
        stageResults.inc(info.getCommonName(), event.getType().name().toLowerCase());
    }

    private static void updateStageTime(StageInfo info, Event event) {
        boolean success = event.isSuccess();

        MetricName metricName = metricNameForSize(extractIncomingFileSizeFromEvent(event))
                .getOrElse(new MetricName());

        metricName = new MetricName(Cf.list(info.getCommonName()).plus(metricName.asList()));
        stages.update(new MeasureInfo(event.getDuration(), success), metricName, UpdateMode.RECURSIVE);
    }

    private static Option<DataSize> extractIncomingFileSizeFromEvent(Event event) {
        if (event.getResultO().isPresent()) {
            return extractIncomingFileSize(event.getResultO().get());
        } else {
            return Option.empty();
        }
    }

    private static Option<DataSize> extractIncomingFileSize(Object result) {
        if (result instanceof Option) {
            if (((Option) result).isPresent()) {
                return extractIncomingFileSize(((Option) result).get());
            } else {
                return Option.empty();
            }
        } if (result instanceof IncomingFile) {
            return ((IncomingFile) result).getContentLength();
        } else if (result instanceof ServiceIncomingFile) {
            return ((ServiceIncomingFile) result).getIncomingFile().getContentLength();
        } else if (result instanceof ServiceIncomingImage) {
            return ((ServiceIncomingImage) result).getIncomingFile().getContentLength();
        }
        return Option.empty();
    }

    private static Option<MetricName> metricNameForSize(Option<DataSize> contentLength) {
        for (DataSize dataSize : contentLength) {
            if (dataSize.le(DataSize.fromMegaBytes(1))) {
                return Option.of(new MetricName("size < 1 Mb"));
            } else if (dataSize.le(DataSize.fromMegaBytes(10))) {
                return Option.of(new MetricName("1Mb < size < 10Mb"));
            } else if (dataSize.le(DataSize.fromMegaBytes(100))) {
                return Option.of(new MetricName("10Mb < size < 100Mb"));
            } else {
                return Option.of(new MetricName("100Mb < size"));
            }
        }
        return Option.empty();
    }
}
