package ru.yandex.chemodan.uploader.log;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * This log used for calculating statistic about file upload
 * (look at CHEMODAN-14287 for details)
 * @author akirakozov
 */
public class UploadStatisticLogger {
    private static final Logger logger = LoggerFactory.getLogger(UploadStatisticLogger.class);

    public static void logUploadEvent(MpfsRequestRecord record, PutResult result, Option<String> userAgent) {
        try {
            Check.C.isTrue(record.getStatus().waitForExternalField().isPresent(),
                    "Expected uploading to disk record");

            logger.info(formatLogMessage(record, result, userAgent, new DateTime(DateTimeZone.getDefault())));
        } catch (Exception ex){
            logger.warn("Couldn't log upload event " + ex.getMessage(), ex);
        }
    }

    static String formatLogMessage(MpfsRequestRecord record, PutResult result,
            Option<String> userAgent, DateTime now)
    {
        ChemodanFile chemodanFile = record.getRequest().chemodanFile;
        Option<IncomingFile> inFile = record.getStatus().waitForExternalField().get().get().getResultO();
        DataSize contentLength = inFile.map(IncomingFile.getContentLengthF())
                .getOrElse(Option.empty()).getOrElse(DataSize.ZERO);

        MapF<String, String> data = Cf.linkedHashMap();
        data.put("id", record.meta.id.id);
        data.put("tskv_format", "ydisk-uploader-stat-log");
        data.put("type", record.getClass().getSimpleName());
        data.put("timestamp", StatTimeFormatterUtils.TIMESTAMP_FORMATTER.print(now));
        data.put("timezone", StatTimeFormatterUtils.TIMEZONE_FORMATTER.print(now));
        data.put("unixtime", String.valueOf(now.getMillis() / 1000));
        data.put("puid", chemodanFile.getUidOrSpecial().toSpecialOrUidString());
        data.put("path", chemodanFile.getPath());
        data.put("content_length", contentLength.toString());
        data.put("user-agent", userAgent.getOrElse(""));
        data.put("status", String.valueOf(result.getStatusCode()));

        return TskvUtils.formatTskvLine(data);
    }
}
