package ru.yandex.chemodan.uploader.mime;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.mime.MimeTypeCorrector;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ExecUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.mime.detect.MimeTypeDetector;
import ru.yandex.misc.time.Stopwatch;

/**
 * @author bursy
 */
@RequiredArgsConstructor
public class FfprobeMimeTypeCorrector implements MimeTypeCorrector {
    private static final Logger logger = LoggerFactory.getLogger(FfprobeMimeTypeCorrector.class);

    private static final String MIME_UNKNOWN = "application/octet-stream";

    private final String ffprobePath;
    private final Duration timeout;

    private final DynamicProperty<Boolean> correctionEnabled =
            new DynamicProperty<>("ffprobe-mime-type-corrector-enabled", false);

    @Override
    public boolean isApplicable(String detectedMimeType) {
        return correctionEnabled.get() && MIME_UNKNOWN.equals(detectedMimeType);
    }

    @Override
    public Option<String> correct(File2 file, Option<String> filename) {
        try {
            logger.debug("Running ffprobe mime type corrector for {}", file);
            Stopwatch sw = Stopwatch.createAndStart();

            // some files (for example, videos with service_provider: rubetek) are missing magic bytes whatsoever
            // they are still valid so we attempt to perform a very fast check with ffprobe to verify that
            ExecResult execResult = ExecUtils.executeGrabbingOutput(createCommand(file), (int) timeout.getMillis());

            sw.stop();
            logger.debug("ffprobe mime type corrector results for {}, took {}, isSuccess={}", file, sw.millisDuration(),
                    execResult.isSuccess());

            if (execResult.isSuccess()) {
                // a bit naive, but ffprobe returned at least something - trusting it
                return MimeTypeDetector.detect(filename, Option.empty());
            }
        } catch (Exception e) {
            logger.warn("ffprobe mime type corrector failed", e);
        }

        return Option.empty();
    }

    private ListF<String> createCommand(File2 file) {
        return Cf.list(ffprobePath, file.getAbsolutePath());
    }
}
