package ru.yandex.chemodan.uploader.mulca;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import org.joda.time.Instant;

import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author bursy
 */
public class SimultaneousMulcaUploadLimitedThroughputStream extends InputStreamX {
    private final long totalLength;
    private final IncomingFile incomingFile;
    private final double uploadedToDownloadedRatio;
    private final Instant userUploadStartTime;

    private long totalUploaded;

    public SimultaneousMulcaUploadLimitedThroughputStream(long totalLength, IncomingFile incomingFile,
            double uploadedToDownloadedRatio, Instant userUploadStartTime)
    {
        super(incomingFile.getRawFile().getInputUnchecked());
        this.totalLength = totalLength;
        this.incomingFile = incomingFile;
        this.uploadedToDownloadedRatio = uploadedToDownloadedRatio;
        this.userUploadStartTime = userUploadStartTime;
    }

    @Override
    public int read() throws IOException {
        sleepIfNeeded();

        int c = super.read();
        if (c >= 0) {
            ++this.totalUploaded;
        }
        return c;
    }

    @Override
    public int read(byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        sleepIfNeeded();

        int nread = super.read(b, off, len);
        if (nread >= 0) {
            this.totalUploaded += nread;
        }
        return nread;
    }

    @Override
    public long skip(long n) throws IOException {
        sleepIfNeeded();

        long skipped = super.skip(n);
        if (skipped >= 0) {
            this.totalUploaded += skipped;
        }
        return skipped;
    }

    private void sleepIfNeeded() {
        long downloaded = incomingFile.getRawFile().length();

        if (downloaded == totalLength) {
            // full file is already downloaded, limits are off
            return;
        }

        if (uploadedToDownloadedRatio * downloaded > totalUploaded) {
            // already behind schedule, no need to slow down upload
            return;
        }

        // general idea: if we're ahead on upload, need to wait for download to catchup to given ratio

        // UC = uploaded currently
        // DC = downloaded currently
        // DT = downloaded target
        // TC = time current
        // TT = time target
        // k = target ratio
        // DS = download speed
        // TX = duration to sleep

        // ratio: DT * k = UC

        // TX = TT - TC
        // DT - DC = (TT - TC) * DS
        // TX = (DT - DC) / DS = (UC / k - DC) / DS

        // speed is estimated ~constant here, the ratio is here to give room in case of speed spikes
        Instant now = new Instant();
        long downloadTimeMillis = now.getMillis() - userUploadStartTime.getMillis();
        if (downloadTimeMillis <= 0) {
            // this should never happen(tm)
            // upload started instantly, so we should be able to get ahead on upload for a couple of reads
            return;
        }

        // bytes/millis
        double downloadSpeed = (double) downloaded / downloadTimeMillis;

        long timeToSleepMillis = (long) ((totalUploaded / uploadedToDownloadedRatio - downloaded) / downloadSpeed);

        if (timeToSleepMillis > 0) {
            ThreadUtils.sleep(timeToSleepMillis, TimeUnit.MILLISECONDS);
        }
    }
}
