package ru.yandex.chemodan.uploader.mulca;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import lombok.RequiredArgsConstructor;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.registry.Stages;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecordUtils;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author bursy
 */
@RequiredArgsConstructor
public class SimultaneousMulcaUploadManager extends DelayingWorkerServiceBeanSupport {
    private static final Logger logger = LoggerFactory.getLogger(SimultaneousMulcaUploadManager.class);

    private final ExecutorService executorService;
    private final Stages stages;
    private final UploadRegistry<MpfsRequestRecord> uploadRegistry;
    private final BleedingEdge bleedingEdge;

    private final DynamicProperty<Double> minimalDownloadedRatioToStart =
            DynamicProperty.cons("uploader-simultaneous-mulca-minimal-downloaded-ratio-to-start", 0.5);
    private final DynamicProperty<Long> estimatedMulcaUploadSpeedBps =
            DynamicProperty.cons("uploader-simultaneous-mulca-upload-estimated-speed-bps", 50000000L);
    private final DynamicProperty<Double> uploadedToDownloadedRatio =
            DynamicProperty.cons("uploader-simultaneous-mulca-uploaded-to-downloaded-ratio", 0.8);
    private final DynamicProperty<Long> minimalFileSize =
            DynamicProperty.cons("uploader-simultaneous-mulca-upload-min-size-bytes", 104857600L);

    private final MapF<UploadRequestId, CompletableFuture<MulcaUploadInfo>> results = Cf.concurrentHashMap();

    public void tryToSubmitUpload(Record<MpfsRequestRecord.UploadToDefault> record) {
        try {
            if (results.getO(record.get().meta.id).isPresent()) {
                return;
            }

            IncomingFile incomingFile = getAndValidateIncomingFile(record);
            Option<Instant> userUploadStartTimeO = getUserUploadStartTime(record);

            if (!userUploadStartTimeO.isPresent()) {
                // definitely too early, user upload didn't start yet
                return;
            }

            if (shouldSubmitUpload(incomingFile, userUploadStartTimeO.get())) {
                submitUpload(record, stages.uploadFileToMulcaF(
                        prepareInputStream(incomingFile, userUploadStartTimeO.get()),
                        record.get().getRequest().chemodanFile.getUidOrSpecial()));
            }
        } catch (IllegalStateException e) {
            // ignore the invalid records to avoid spamming logs
            return;
        } catch (Throwable e) {
            logger.error("Unable to submit record for simultaneous upload", e);
        }
    }

    private boolean shouldSubmitUpload(IncomingFile incomingFile, Instant userUploadStartTime) {
        long receivedFromUser = incomingFile.getRawFile().length();

        // CL = content length (total length), bytes
        long contentLength = incomingFile.getContentLength().map(DataSize::toBytes).get();

        // point at which we've downloaded % of file from user to always have some data at start
        long downloadTargetByDownloadedRatio = (long) (minimalDownloadedRatioToStart.get() * contentLength);
        if (receivedFromUser < downloadTargetByDownloadedRatio) {
            return false;
        }

        Instant now = new Instant();
        long downloadTimeMillis = now.getMillis() - userUploadStartTime.getMillis();
        if (downloadTimeMillis <= 0) {
            // this should never happen(tm), obviously too early
            return false;
        }

        // DS = download speed (from user), bytes/second
        double downloadSpeed = (double) receivedFromUser / downloadTimeMillis * 1000;
        // US = upload speed (to mulca), bytes/second
        long uploadSpeed = estimatedMulcaUploadSpeedBps.get();
        // estimate that upload & download will finish at same time at current speeds
        // DT = download target, amount of bytes when it's estimated upload should start
        // (CL - DT) / DS = CL / US
        // DT = CL * (1 - DS / US)
        long downloadTarget = (long) (contentLength * (1 - downloadSpeed / uploadSpeed));

        return receivedFromUser >= downloadTarget;
    }

    private void submitUpload(Record<MpfsRequestRecord.UploadToDefault> record,
            Function0<MulcaUploadInfo> uploadFileToMulcaF)
    {
        UploadRequestId id = record.get().meta.id;

        logger.info("Submitting simultaneous upload for id={}", id);

        CompletableFuture<MulcaUploadInfo> future = new CompletableFuture<>();
        executorService.submit(() -> {
            logger.info("Starting simultaneous upload for id={}", id);

            MulcaUploadInfo mulcaUploadInfo;
            try {
                mulcaUploadInfo = uploadFileToMulcaF.apply();
            } catch (Throwable e) {
                logger.warn("Simultaneous upload failed for id={}", id, e);
                future.completeExceptionally(e);
                throw e;
            }

            if (recheckValidUpload(record)) {
                logger.info("Simultaneous upload successfully completed for id={}, mulcaId={}", id,
                        mulcaUploadInfo.getMulcaId());
                future.complete(mulcaUploadInfo);
            } else {
                logger.warn("Throwing away simultaneous upload result for id={}, mulcaId={}", id,
                        mulcaUploadInfo.getMulcaId());
                future.cancel(false);
                cleanUpMulcaId(record.get(), mulcaUploadInfo.getMulcaId());
            }
        });
        results.put(id, future);
    }

    public MulcaUploadInfo awaitAndGetResult(Record<MpfsRequestRecord.UploadToDefault> record,
            Option<Duration> timeout)
    {
        UploadRequestId id = record.get().meta.id;

        Option<CompletableFuture<MulcaUploadInfo>> futureO = results.getO(id);
        if (!futureO.isPresent()) {
            throw new SimultaneousMulcaUploadException("Requested simultaneous upload result is missing");
        }

        MulcaUploadInfo mulcaUploadInfo;
        try {
            mulcaUploadInfo = getFutureResult(futureO.get(), timeout);
        } catch (CancellationException e) {
            throw new SimultaneousMulcaUploadException("Requested simultaneous upload result but upload was canceled", e);
        } catch (Throwable e) {
            throw new SimultaneousMulcaUploadException("Requested simultaneous upload result failed", e);
        }

        results.removeTs(id);

        return mulcaUploadInfo;
    }

    public boolean isUploadInProgress(UploadRequestId id) {
        return results.getO(id).isPresent();
    }

    public void cancelUpload(UploadRequestId id) {
        Option<CompletableFuture<MulcaUploadInfo>> futureO = results.removeO(id);

        if (!futureO.isPresent()) {
            return;
        }

        // first, attempt to clean up id directly if it was already completed
        try {
            Option.ofNullable(futureO.get().getNow(null))
                    .forEach(info -> uploadRegistry.findRecord(id)
                            .forEach(record -> cleanUpMulcaId(record, info.getMulcaId())));
        } catch (Exception e) {
            // future was not completed, it's fine
        }

        logger.info("Canceling simultaneous upload for id={}", id);
        // this way the upload will keep running for now but will be discarded on completion
        futureO.get().cancel(false);
    }

    public boolean isUploadSupported(PassportUid uid, DataSize contentLength) {
        return bleedingEdge.isOnBleedingEdge(uid) && contentLength.toBytes() > minimalFileSize.get();
    }

    private InputStreamSource prepareInputStream(IncomingFile incomingFile, Instant userUploadStartTime) {
        Option<Long> contentLengthO = incomingFile.getContentLength().map(DataSize::toBytes);

        return new InputStreamSource() {
            @Override
            public InputStream getInput() throws IOException {
                return new SimultaneousMulcaUploadLimitedThroughputStream(contentLengthO.get(), incomingFile,
                        uploadedToDownloadedRatio.get(), userUploadStartTime);
            }

            @Override
            public Option<Long> lengthO() throws RuntimeIoException {
                return contentLengthO;
            }
        };
    }

    private <T> T getFutureResult(CompletableFuture<T> future, Option<Duration> timeout) {
        try {
            return timeout.isPresent()
                   ? future.get(timeout.get().getMillis(), TimeUnit.MILLISECONDS)
                   : future.get();
        } catch (TimeoutException | InterruptedException | ExecutionException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private boolean recheckValidUpload(Record<MpfsRequestRecord.UploadToDefault> record) {
        try {
            return recheckValidUploadUnsafe(record);
        } catch (Throwable e) {
            logger.warn("Simultaneous upload check failed", e);
            return false;
        }
    }

    private boolean recheckValidUploadUnsafe(Record<MpfsRequestRecord.UploadToDefault> record) {
        UploadRequestId id = record.get().meta.id;

        // basic check that we even tracked the record
        if (!isUploadInProgress(id)) {
            logger.warn("Simultaneous upload check failed: upload finished but id was not tracked, id={}", id);
            return false;
        }

        IncomingFile incomingFile = getAndValidateIncomingFile(record);
        long contentLength = incomingFile.getContentLength().map(DataSize::toBytes).get();
        long localLength = incomingFile.getRawFile().length();

        // naive check that uploaded file didn't differ
        if (contentLength != localLength) {
            logger.warn("Simultaneous upload check failed: lengths differ, contentLength={}, localLength={}, id={}",
                    contentLength, localLength, id);
            return false;
        }

        // check for manual failure on record level
        MpfsRequestRecord.UploadToDefault freshRecord = record.refreshAndGet();
        if (freshRecord.getStatus().simultaneousMulcaUploadInfo.get().isPermanentFailure()) {
            logger.warn("Simultaneous upload check failed: state already set to failed, id={}", id);
            return false;
        }

        return true;
    }

    private void cleanUpMulcaId(MpfsRequestRecord record, MulcaId mulcaId) {
        Option<String> yandexCloudRequestId = record.getRequest().yandexCloudRequestId;

        MpfsRequest.MarkMulcaIdsForRemove request =
                new MpfsRequest.MarkMulcaIdsForRemove(ApiVersion.V_0_2, Cf.list(mulcaId), yandexCloudRequestId);
        uploadRegistry.saveRecord(
                MpfsRequestRecordUtils.consF(request, RequestRevision.initial(HostInstant.hereAndNow())));
    }

    private IncomingFile getAndValidateIncomingFile(Record<?> record) {
        MutableState<IncomingFile> userFileState = getUserFileState(record);

        Option<IncomingFile> incomingFileO = userFileState.get().getIntermediateOrFinalResultO();
        Check.notEmpty(incomingFileO, "Record has no incoming file");

        Check.notEmpty(incomingFileO.get().getContentLength(), "Record has no content length");

        return incomingFileO.get();
    }

    private Option<Instant> getUserUploadStartTime(Record<?> record) {
        MutableState<IncomingFile> userFileState = getUserFileState(record);

        return userFileState.get().asInProgress().map(State.InProgress::getStarted);
    }

    private MutableState<IncomingFile> getUserFileState(Record<?> record) {
        Option<MutableState<IncomingFile>> userFileStateO = record.get().getStatus().waitForExternalField();
        Check.notEmpty(userFileStateO, "Record has no external field");

        return userFileStateO.get();
    }

    private void cleanUpInvalidRecords() {
        ListF<UploadRequestId> aliveRecordsIds =
                uploadRegistry.findRecordsInProgress(Option.empty()).map(record -> record.meta.id);

        ListF<UploadRequestId> invalidRecordIds = results.keys().filterNot(aliveRecordsIds::containsTs);

        logger.info("Cleaning up invalid records, marking them as canceled");
        invalidRecordIds.forEach(this::cancelUpload);
    }

    @Override
    public void stop() {
        super.stop();
        executorService.shutdown();
    }

    @Override
    protected void execute() {
        cleanUpInvalidRecords();
    }

    @Override
    protected Duration defaultDelay() {
        return Duration.standardHours(1);
    }
}
