package ru.yandex.chemodan.uploader.office;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.mpfs.MpfsUtils;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.registry.StageListenerPoolWithChildren;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecordUtils;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.stage.LoggableStageProcessor;
import ru.yandex.chemodan.uploader.web.UploadTimeoutHolder;
import ru.yandex.chemodan.uploader.web.data.util.UploadServletUtils;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.local.queue.LocalQueuePush;
import ru.yandex.commune.uploader.registry.CallbackResponse;
import ru.yandex.commune.uploader.registry.RecordWrapper;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.web.data.DiskWritePolicy;
import ru.yandex.commune.uploader.web.data.HttpPutRequestContext;
import ru.yandex.commune.uploader.web.data.RangedPutUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;

/**
 * @author akirakozov
 */
public class OfficeUploadManager {

    @Autowired
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    private LocalQueuePush localQueuePush;
    @Autowired
    private DiskWritePolicy diskWritePolicy;
    @Autowired
    private StageListenerPoolWithChildren<Record<?>> stageListenerPool;
    @Autowired
    private UploadTimeoutHolder uploadTimeoutHolder;

    @Autowired
    private MpfsClient mpfsClient;

    public OfficeUploadResult startUpload(String resourceId, String accessToken,
            String accessTokenTtl, MapF<String, String> headers, Option<String> userAgent,
            HttpPutRequestContext context, Option<String> ycrid)
    {
        MpfsOfficeStoreResponseInfo r = initOfficeUpload(resourceId, accessToken, accessTokenTtl, headers);
        int statusCode = r.statusCode;
        Option<UploadRequestId> requestId = Option.empty();
        if (r.storeInfo.isPresent()) {
            MpfsInitialStoreInfo info = r.storeInfo.get();
            MpfsRequestRecord record = saveRecord(info, ycrid);

            PutResult result = UploadServletUtils.handleAndExecuteAsap(
                    record, localQueuePush, stageListenerPool,
                    uploadTimeoutHolder.getWaitCommitFileUploadTimeout(true, true),
                    processUploadF(RecordWrapper.cons(uploadRegistry, record), context), userAgent);
            statusCode = rewriteStatusCodeIfNecessary(statusCode, result);

            requestId = Option.of(record.meta.id);
        }
        return new OfficeUploadResult(statusCode, r.body, r.headers, requestId);
    }

    private int rewriteStatusCodeIfNecessary(int statusCode, PutResult result) {
        switch (result) {
            case COMPLETED:
                // On success return mpfs status code
                return statusCode;
            case PARTIAL:
                // Office doesn't support partial upload,
                // so we should override this success status with fail
                return HttpStatus.SC_500_INTERNAL_SERVER_ERROR;
            default:
                // Override mpfs status code for failures
                return result.getStatusCode();
        }
    }

    private MpfsOfficeStoreResponseInfo initOfficeUpload(
            String resourceId, String accessToken, String accessTokenTtl, MapF<String, String> headers)
    {
        String taskName = MpfsRequest.UploadToDefault.class.getSimpleName();
        CallbackResponse response = LoggableStageProcessor.processStage(taskName, "init-office-store",
                () -> MpfsUtils.convert(
                        mpfsClient.getOfficeStoreInfo(resourceId, accessToken, accessTokenTtl, headers)));
        MpfsOfficeStoreResponseInfo info = LoggableStageProcessor.processStage(taskName, "parse-response",
                () -> getMpfsInitialStoreInfo(response.getResponse()));

        return info;
    }

    private MpfsOfficeStoreResponseInfo getMpfsInitialStoreInfo(String response) {
        return MpfsOfficeStoreResponseInfo.PS.getParser().parseJson(response);
    }

    private MpfsRequestRecord saveRecord(MpfsInitialStoreInfo info, Option<String> ycrid) {
        ChemodanFile chemodanFile = ChemodanFile.cons(
                UidOrSpecial.uid(new PassportUid(info.uid)),
                info.fileId,
                info.path);

        return uploadRegistry.saveRecord(MpfsRequestRecordUtils.consF(
                new MpfsRequest.UploadToDefault(
                        ApiVersion.valueOf(info.apiVersion),
                        chemodanFile,
                        Option.of(UrlUtils.uri(info.callback)),
                        Option.empty(),
                        // TODO: Generate ycrid if it's empty
                        ycrid),
                RequestRevision.initial(HostInstant.hereAndNow())));
    }

    private Function0<PutResult> processUploadF(final RecordWrapper<?> record, HttpPutRequestContext context)
    {
        return () -> RangedPutUtils.processPut(localFileManager, record, context, diskWritePolicy);
    }
}
