package ru.yandex.chemodan.uploader.preview;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.preview.annotations.AlbumAnnotationManager;
import ru.yandex.chemodan.uploader.preview.annotations.Annotation;
import ru.yandex.chemodan.util.ExecUtils2;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ShellCommandBuilder;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author metal
 */
public class AlbumPreviewManager {
    @Autowired
    private FoggingManager foggingManager;

    @Autowired
    private AlbumAnnotationManager albumAnnotationManager;

    @Value("${imagemagick.convert}")
    private String convertCmd;

    @Value("${imagemagick.composite}")
    private String compositeCmd;

    @Value("${uploader.album.preview.fog.color}")
    private String fogColor;

    @Value("${uploader.album.preview.fog.intensity}")
    private int fogIntensity;

    @Value("${disk.logo.width}")
    private int logoWidth;

    @Value("${disk.logo.height}")
    private int logoHeight;

    @Value("${uploader.album.preview.default.width}")
    private int previewDefaultWidth;

    @Value("${uploader.album.preview.default.height}")
    private int previewDefaultHeight;

    @Value("${uploader.album.preview.annotation.font}")
    private String font;

    @Value("${uploader.album.preview.annotation.font.color}")
    private String fontColor;

    @Value("${uploader.album.preview.annotation.name.offset}")
    private int nameOffset;

    @Value("${uploader.album.preview.annotation.name.size}")
    private int namePointSize;

    @Value("${uploader.album.preview.annotation.user.offset}")
    private int userOffset;

    @Value("${uploader.album.preview.annotation.user.size}")
    private int userPointSize;

    @Value("${uploader.preview.album.timeout}")
    private Duration generatePreviewTimeout;

    public PreviewInfo generateAlbumPreview(File2 image, String albumName, String userName) {
        Dimension needDimension = new Dimension(previewDefaultWidth, previewDefaultHeight);
        Dimension origSize = preparePreviewImage(image, needDimension);

        return ThreadLocalTimeout.withTimeout(generatePreviewTimeout, () -> {
            foggingManager.fogImageWithColor(image, fogColor, fogIntensity);

            Annotation albumNameAnnotation = new Annotation(font, fontColor, 0, nameOffset, namePointSize, albumName);
            Annotation userAnnotation = new Annotation(font, fontColor, 0, userOffset, userPointSize, userName);

            File2.withNewTempDir("album-preview", tempDir -> {
                File2 albumAnnotation = albumAnnotationManager.createAlbumAnnotation(tempDir,
                        new Dimension(logoWidth, logoHeight),
                        albumNameAnnotation, userAnnotation);

                ListF<String> cmd = Cf.list(compositeCmd,
                        "-gravity", WatermarkPosition.CENTER.toImageMagickFormat(),
                        albumAnnotation.getAbsolutePath(),
                        image.getAbsolutePath(), // source
                        "png:" + image.getAbsolutePath()); // result

                ExecUtils2.executeWithLogging(cmd);
            });

            return new PreviewInfo(ImageFormat.PNG, Option.of(origSize), needDimension, Option.empty());
        });
    }

    private Dimension preparePreviewImage(File2 image, Dimension needDimension) {
        return ThreadLocalTimeout.withTimeout(generatePreviewTimeout, () -> {
            ShellCommandBuilder commandBuilder = new ShellCommandBuilder();
            commandBuilder.append(convertCmd,
                    image.getAbsolutePath(), // source
                    "-print", "%wx%h");
            commandBuilder.append(ImageMagick.fillResizeCmd(needDimension));
            commandBuilder.append(image.getAbsolutePath()); // result

            ExecResult result = ExecUtils2.executeGrabbingOutputWithLogging(commandBuilder.commands());
            return Dimension.valueOf(result.getStdout().trim());
        });
    }
}
