package ru.yandex.chemodan.uploader.preview;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.commune.image.RotateAngle;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.commune.image.imageMagick.ImageMagickConfiguration;
import ru.yandex.commune.image.imageMagick.ImageMagickTargetInfo;
import ru.yandex.commune.image.imageMagick.resize.CropAndResizeOperation;
import ru.yandex.commune.image.imageMagick.resize.ResizeOperation;
import ru.yandex.commune.image.imageMagick.resize.ResizeProportionalMinimizeOnly;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.mime.detect.MimeTypeDetector;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @url https://svn.yandex.ru/websvn/wsvn/fotki/fotki/trunk/common/src/main/java/ru/yandex/fotki/outside/imageMagick/ImageMagickReportParser.java
 * @author akirakozov
 * @author ssytnik
 */
public class PreviewImageManager {
    private static final Logger logger = LoggerFactory.getLogger(PreviewImageManager.class);

    @Value("${preview.image.one.enabled}")
    private boolean previewImageOneEnabled;

    @Value("${preview.image.imagemagick.ping.timeout}")
    private Duration imageMagickPingTimeout;
    @Value("${uploader.imagemagick.max.disk.size}")
    private DataSize maxDiskSize;
    @Value("${uploader.imagemagick.max.map.size}")
    private DataSize maxMapSize;
    @Value("${uploader.imagemagick.max.memory.size}")
    private DataSize maxMemorySize;
    @Value("${uploader.imagemagick.max.area.size}")
    private DataSize maxAreaSize;

    @Autowired
    private Dcraw dcraw;
    @Autowired
    private HeifConverter heifConverter;

    private static final SetF<String> SVG_MIME_TYPES = Cf.set("image/svg+xml");

    private static final SetF<String> HEIF_MIME_TYPES = Cf.set("image/heif", "image/heic");

    private static final SetF<String> RAW_MIME_TYPES = Cf.set(
            "image/nef",
            "image/x-nikon-nef",
            "image/x-canon-cr2",
            "image/cr2",
            "image/x-canon-raw",
            "image/canon raw",
            "image/arw",
            "image/x-sony-raw",
            "image/x-sony-arw",
            "image/x-olympus-orf");

    private static final SetF<String> PSD_MIME_TYPES = Cf.set(
            "image/vnd.adobe.photoshop",
            "image/psd",
            "image/x-photoshop",
            "application/psd",
            "application/x-photoshop");

    private static final SetF<String> JPEG_MIME_TYPES = Cf.set(
            "image/jpeg",
            "image/jpg");

    private static final SetF<String> SUPPORTED_MIME_TYPES = Cf.set(
            "image/pjpeg",
            "image/png",
            "image/gif",
            "image/tiff",
            "image/bmp",
            "image/x-ms-bmp",
            "image/pict",
            "image/jps",
            "image/x-portable-greymap",
            "image/x-portable-pixmap",
            "image/x-png",
            "image/x-bmp",
            "image/x-icon",
            "image/vnd.microsoft.icon",
            "image/ico",
            "image/jp2",
            "image/x-adobe-dng",
            "application/ico",
            "application/x-ico")
            .plus(JPEG_MIME_TYPES)
            .plus(RAW_MIME_TYPES)
            .plus(PSD_MIME_TYPES)
            .plus(HEIF_MIME_TYPES);

    // TODO: move to properties?
    private static final Duration DEFAULT_TIMEOUT = Duration.standardMinutes(5);

    public static boolean isSupportedMimeType(String mimeType) {
        Option<String> normalized = MimeTypeDetector.normalizeMimeType(Option.of(mimeType));

        return normalized.isMatch(SUPPORTED_MIME_TYPES::containsTs) || normalized.isMatch(type -> type.startsWith("image/"));
    }

    public static boolean isSvg(String mimeType) {
        return MimeTypeDetector.normalizeMimeType(Option.of(mimeType)).isMatch(SVG_MIME_TYPES::containsTs);
    }

    private boolean isRawFormat(String normalizedMimeType) {
        return RAW_MIME_TYPES.containsTs(normalizedMimeType);
    }

    public static boolean isPsd(String normalizedMimeType) {
        return PSD_MIME_TYPES.containsTs(normalizedMimeType);
    }

    private boolean isHeif(String normalizedMimeType) {
        return HEIF_MIME_TYPES.containsTs(normalizedMimeType);
    }

    public PreviewInfo generateOnePreviewForImage(File2 localFile, Option<String> contentType, Dimension size,
            boolean crop, String previewName, Option<Integer> quality)
    {
        localFile = prepareFileForPreview(localFile, contentType);

        ImageMagickTargetInfo targetInfo = getImageMagickTargetInfo(size, crop, previewName);
        ImageMagick imageMagick = configureImageMagick(quality);

        String report = imageMagick.generateThumbnails(localFile, Cf.list(targetInfo), RotateAngle.AUTO, false, false);

        return new ConvertReportParser().parseReport(report, targetInfo.getFileName());
    }

    private File2 prepareFileForPreview(File2 localFile, Option<String> contentType) {
        if (contentType.isPresent()) {
            String normalizedMimeType = MimeTypeDetector.normalizeMimeType(contentType).getOrElse("");
            if (isRawFormat(normalizedMimeType)) {
                logger.info("File is RAW image. Extract preview from it's content");
                localFile = dcraw.extractPreviewFromRawImage(localFile);
            } else if (isHeif(normalizedMimeType)) {
                // there's a bug in mobile autoupload resulting in jpg files with .heic extension, so we doublecheck
                // see https://st.yandex-team.ru/CHEMODAN-40056
                Option<String> actualMimeType = MimeTypeDetector.detect(Option.empty(), Option.of(localFile));
                if (actualMimeType.isMatch(JPEG_MIME_TYPES::containsTs)) {
                    return localFile;
                }

                logger.info("File is Heif. Use heif converter");
                localFile = heifConverter.convertHeifToJpeg(localFile);
            } else if (isPsd(normalizedMimeType)) {
                logger.info("File is PSD. Use 0-layer for preview");
                localFile = new File2(localFile.getAbsolutePath() + "[0]");
            }
        }
        return localFile;
    }

    private ImageMagickTargetInfo getImageMagickTargetInfo(Dimension size, boolean crop, String previewName) {
        ResizeOperation resizeOperation = crop ?
                new CropAndResizeOperation(size) :
                new ResizeProportionalMinimizeOnly(size);
        return new ImageMagickTargetInfo(resizeOperation, previewName);
    }

    private ImageMagick configureImageMagick(Option<Integer> quality) {
        Duration timeout = ThreadLocalTimeout.timeout().getOrElse(DEFAULT_TIMEOUT);
        ImageMagickConfiguration.Builder confBuilder = ImageMagickConfiguration
                .custom()
                .setPingTimeout(imageMagickPingTimeout)
                .setStillImageConvertTimeout(timeout)
                .setAnimationConvertTimeout(timeout)
                .setMaxAreaSize(maxAreaSize)
                .setMaxMapSize(maxMapSize)
                .setMaxMemorySize(maxMemorySize)
                .setMaxDiskSize(maxDiskSize);

        if (quality.isPresent()) {
            confBuilder
                    .setJpegQualityPercent(quality.get())
                    .setPngQualityPercent(quality.get());
        }
        return new ImageMagick(confBuilder.build());
    }

    public boolean isEnabledForOne() {
        return previewImageOneEnabled;
    }
}
