package ru.yandex.chemodan.uploader.processor;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.installer.InstallerModifierParams;
import ru.yandex.chemodan.uploader.log.OauthCodeResult;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.preview.PreviewInfo;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.Stages;
import ru.yandex.chemodan.uploader.registry.record.status.DigestCalculationStatus;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateAlbumPreviewResult;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.stage.StageProcessor;
import ru.yandex.commune.uploader.registry.StageResult;
import ru.yandex.commune.uploader.registry.StageResult.StageResultType;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class SyncRequestProcessor {
    private static final Logger logger = LoggerFactory.getLogger(SyncRequestProcessor.class);

    private static final String EXTRACT_EXIF_TASK = "SyncExtractExif";
    private static final String CHECK_VIRUSES_TASK = "SyncCheckViruses";
    private static final String PATCH_INSTALLER_TASK = "PatchInstaller";
    private static final String RECALC_DIGESTS_TASK = "SyncRecalcDigests";
    private static final String EXTRACT_ROTATION_TASK = "SyncExtractRotation";
    private static final String GENERATE_PREVIEW_TASK = "GeneratePreview";
    private static final String GENERATE_DYNAMIC_ALBUM_PREVIEW_TASK = "GenerateDynamicAlbumPreview";
    private static final String GENERATE_STATIC_ALBUM_PREVIEW_TASK = "GenerateStaticAlbumPreview";

    private final Stages stages;
    private final StageProcessor stageProcessor;

    public static class ExtractedExifData {
        static final ExtractedExifData EMPTY =
                new ExtractedExifData(MulcaId.fromSerializedString("none"), "[{}]");

        public final String exifInJson;
        public final MulcaId mulcaId;

        public ExtractedExifData(MulcaId exifMulcaId, String exifInJson) {
            this.exifInJson = exifInJson;
            this.mulcaId = exifMulcaId;
        }
    }

    public static class DigestsRecalcResult {

        public final ContentInfo info;
        public final Option<MulcaId> mulcaId;
        public final Option<ExifInfo> exif;
        public final Option<AntivirusResult> antivirus;

        private DigestsRecalcResult(ContentInfo info,
                Option<MulcaId> mulcaId,
                Option<ExifInfo> exif,
                Option<AntivirusResult> antivirus)
        {
            this.info = info;
            this.mulcaId = mulcaId;
            this.exif = exif;
            this.antivirus = antivirus;
        }
    }

    public SyncRequestProcessor(Stages stages, StageProcessor stageProcessor) {
        this.stages = stages;
        this.stageProcessor = stageProcessor;
    }

    private <T> T withDownloadedFile(String task, final MulcaId sourceFileMulcaId, Function<File2, T> function) {
        return File2.withNewTempFile(function.compose(dest -> process(task, "download-source",
            stages.downloadFromMulcaF(sourceFileMulcaId, dest)).getLocalFile()));
    }

    private Option<ExifInfo> extractExif(String task, File2 downloaded, String contentType) {
        return process(task, "extract-exif-info", stages.tryExtractExifF(downloaded, contentType));
    }

    private Option<ExtractedExifData> extractExifAndUploadToMulca(String task, File2 downloaded) {
        String exif = process(task, "extract-exif", stages.extractFullExifInJsonF(downloaded));

        if (StringUtils.isEmpty(exif)) {
            return Option.empty();
        } else {
            MulcaId exifMulcaId = process(task, "upload-exif-to-mulca",
                    stages.uploadStringToMulcaF(exif, UidOrSpecial.special("exif"))).getMulcaId();

            return Option.of(new ExtractedExifData(exifMulcaId, exif));
        }
    }

    public ExtractedExifData extractExifAndUploadToMulca(final MulcaId sourceFileMulcaId) {
        return withDownloadedFile(EXTRACT_EXIF_TASK, sourceFileMulcaId, downloaded ->
                extractExifAndUploadToMulca(EXTRACT_EXIF_TASK, downloaded)
                        .getOrElse(ExtractedExifData.EMPTY));
    }

    private AntivirusResult checkWithAntivirus(String task, File2 downloaded) {
        StageResult<AntivirusResult> result = process(
                task, "check-by-antivirus",
                stages.checkWithAntivirusF(Option.empty(), downloaded));

        Validate.isTrue(result.getType() == StageResultType.SUCCESS,
                "Unsuccess result type: " + result.getType());

        return result.getResult();
    }

    public AntivirusResult checkWithAntivirus(final MulcaId sourceFileMulcaId) {
        return withDownloadedFile(CHECK_VIRUSES_TASK, sourceFileMulcaId, downloaded ->
            checkWithAntivirus(CHECK_VIRUSES_TASK, downloaded));
    }

    private MulcaId recalcDigests(String task, File2 downloaded) {
        DigestCalculationStatus digestCalculation = process(
                task, "recalc-digest-calculation",
                stages.calculateDigestF(UploadRequestId.valueOf("digest"), downloaded));

        MulcaUploadInfo digestUpload = process(
                task, "recalc-upload-webdav-digest",
                stages.uploadDigestToMulcaF(
                        digestCalculation.getDigestFile().get(), UidOrSpecial.special("digest")));
        return digestUpload.getMulcaId();
    }

    public DigestsRecalcResult recalcDigests(final MulcaId sourceFileMulcaId, boolean withWebdav, boolean withExif, boolean withVirusCheck) {
        return withDownloadedFile(RECALC_DIGESTS_TASK, sourceFileMulcaId, downloaded -> {
                IncomingFile incomingFile = new IncomingFile(
                        Option.empty(), Option.empty(), downloaded);
                ContentInfo info = process(RECALC_DIGESTS_TASK, "recalc-digests",
                        stages.contentInfoF(incomingFile, Option.empty()));

                Option<MulcaId> mulcaId = withWebdav ? Option.of(recalcDigests(RECALC_DIGESTS_TASK, downloaded)) : Option.empty();
                Option<ExifInfo> exifData = withExif ? extractExif(RECALC_DIGESTS_TASK, downloaded, info.getContentType().getOrNull()) : Option.empty();
                Option<AntivirusResult> antivirusResults = withVirusCheck ? Option.of(checkWithAntivirus(RECALC_DIGESTS_TASK, downloaded)) : Option.empty();

                return new DigestsRecalcResult(info, mulcaId, exifData, antivirusResults);
            });
    }

    public int extractRotation(final MulcaId sourceFileMulcaId) {
        return withDownloadedFile(EXTRACT_ROTATION_TASK, sourceFileMulcaId, downloaded ->
            process(EXTRACT_ROTATION_TASK, "extract-rotation", stages.extractRotationF(downloaded)));
    }

    public GenerateImageOnePreviewResult generatePreview(final MulcaId mulcaId, final PreviewSizeParameter size,
            final boolean crop, File2 workingDir, Option<Integer> quality)
    {
        File2 downloaded = workingDir.child("downloaded");
        process(GENERATE_PREVIEW_TASK, "originalFile",
                stages.downloadFromMulcaF(mulcaId, downloaded));

        File2 previewFile = downloaded.sibling("preview");
        PreviewInfo previewInfo = process(GENERATE_PREVIEW_TASK, "generatePreview",
                stages.generateOnePreviewForImageF(downloaded, previewFile.getName(), size, crop, quality));

        return new GenerateImageOnePreviewResult(previewFile, previewInfo);
    }

    public GenerateImageOnePreviewResult generateDynamicAlbumPreview(final MulcaId mulcaId, File2 workingDir,
            final String albumName, final String userName)
    {
        File2 downloaded = workingDir.child("downloaded");
        process(GENERATE_DYNAMIC_ALBUM_PREVIEW_TASK, "originalFile",
                stages.downloadFromMulcaF(mulcaId, downloaded));

        PreviewInfo previewInfo = process(GENERATE_DYNAMIC_ALBUM_PREVIEW_TASK, "generateAlbumPreview",
                stages.generateAlbumPreviewF(downloaded, albumName, userName));

        return new GenerateImageOnePreviewResult(downloaded, previewInfo);
    }

    public GenerateAlbumPreviewResult generateStaticAlbumPreview(final MulcaId mulcaId, File2 workingDir,
            final String albumName, final String userName)
    {
        File2 downloaded = workingDir.child("downloaded");
        process(GENERATE_STATIC_ALBUM_PREVIEW_TASK, "originalFile",
                stages.downloadFromMulcaF(mulcaId, downloaded));

        process(GENERATE_STATIC_ALBUM_PREVIEW_TASK, "generateAlbumPreview",
                stages.generateAlbumPreviewF(downloaded, albumName, userName));

        UidOrSpecial.Special specialUid = UidOrSpecial.special("preview");
        MulcaUploadInfo mulcaUploadInfo = process(GENERATE_STATIC_ALBUM_PREVIEW_TASK, "uploadAlbumPreviewToMulca",
                stages.uploadFileToMulcaF(downloaded, specialUid));

        return new GenerateAlbumPreviewResult(mulcaUploadInfo.getMulcaId());
    }

    public byte[] patchInstaller(MulcaId mulcaId, String clientIp, String cookie, String clientHost,
                                 InstallerModifierParams params, boolean autoLogin)
    {
        byte[] installer = process(PATCH_INSTALLER_TASK, "downloadOriginal",
                stages.downloadFromMulcaToMemoryF(mulcaId)).getContent();
        if (autoLogin) {
            try {
                OauthCodeResult result = process(PATCH_INSTALLER_TASK, "getAuthCode",
                        stages.getAuthCodeForInstaller(clientIp, cookie, clientHost));

                params.setKey(result.getCode());
            } catch (Exception e) {
                logger.error("Failed to get auth code: {}", e);
            }
        }

        process(PATCH_INSTALLER_TASK, "applyPatch",
                stages.patchInstaller(installer, params));

        return installer;
    }

    private <T> T process(String taskName, String stageName, Function0<T> stage) {
        return stageProcessor.process(taskName, stageName, stage);
    }
}
