package ru.yandex.chemodan.uploader.registry;

import org.jetbrains.annotations.NotNull;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.monitoring.Safe;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestMeta;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author vavinov
 * @author akirakozov
 */
public class QueueSensors implements MonicaContainer {

    private final RequestRecordFilter recordFilter = new RequestRecordFilter();
    private final UploadRegistry<MpfsRequestRecord> uploadRegistry;

    private final static Instant startTime = Instant.now();

    private final DynamicProperty<Double> initialDtExpCoef = new DynamicProperty<>("queue-exp-weight-initial-dt-coef", -1 / 1800.0);
    private final DynamicProperty<Double> initialSizeCoef = new DynamicProperty<>("queue-exp-weight-initial-size-coef", 0.0);

    private final DynamicProperty<Double> inProgressDtExpCoef = new DynamicProperty<>("queue-exp-weight-in-progress-dt-coef", -1 / 1800.0);
    private final DynamicProperty<Double> inProgressSizeCoef = new DynamicProperty<>("queue-exp-weight-in-progress-size-coef", 0.0);

    public QueueSensors(UploadRegistry<MpfsRequestRecord> uploadRegistry) {
        this.uploadRegistry = uploadRegistry;
    }

    private ListF<MpfsRequestRecord> inProgress() {
        return Cf.list(uploadRegistry.findRecordsInProgress(Option.empty()));
    }

    private static int secondsToNow(Instant t) {
        return new Duration(t, new Instant()).toStandardSeconds().getSeconds();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int incompleteRequestsCount() {
        return inProgress().size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int incompleteRequestsWaitingForUserCount() {
        return recordFilter.incompleteRequestsWaitingForUser(inProgress()).size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public double exponentialWeightInitial() {
        return recordFilter.incompleteRequestsWaitingForUser(inProgress())
                .map(record -> {
                    final Option<MutableState<IncomingFile>> fieldO = record.getStatus().waitForExternalField();
                    if (!fieldO.isPresent() || fieldO.get().get().isFinished()) {
                        return 0.0;
                    }

                    MutableState<IncomingFile> field = fieldO.get();

                    for (State.Initial<IncomingFile> state : field.get().asInitial()) {
                        Instant whenCreated = record.meta.created;
                        return calculateExpWeight(state, whenCreated, initialDtExpCoef.get(), initialSizeCoef.get());
                    }

                    return 0.0;
                })
                .sum(Cf.Double);
    }

    @NotNull
    private Double calculateExpWeight(State<IncomingFile> state, Instant time, Double dtCoef, Double sizeCoef) {
        Duration pastTime = new Duration(time, Instant.now());
        DataSize dataSize = state.getResultO().flatMapO(IncomingFile::getContentLength).orElse(DataSize.MEGABYTE);

        return Math.exp(pastTime.getStandardSeconds() * dtCoef)
                * (1 + sizeCoef * dataSize.toMegaBytes());
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public double exponentialWeightInProgress() {
        return recordFilter.incompleteRequestsWaitingForUser(inProgress())
                .map(record -> {
                    final Option<MutableState<IncomingFile>> fieldO = record.getStatus().waitForExternalField();
                    if (!fieldO.isPresent() || fieldO.get().get().isFinished()) {
                        return 0.0;
                    }

                    MutableState<IncomingFile> field = fieldO.get();

                    for (State.InProgress<IncomingFile> inProgress : field.get().asInProgress()) {
                        return calculateExpWeight(inProgress, inProgress.getSeenAlive(), inProgressDtExpCoef.get(), inProgressSizeCoef.get());
                    }

                    return 0.0;
                })
                .sum(Cf.Double);
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int incompleteAndUploadedToMulca() {
        return recordFilter.incompleteAndUploadedToMulca(inProgress()).size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int incompleteAndUploadedLocally() {
        return recordFilter.incompleteAndUploadedLocally(inProgress()).size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int incompleteAndUploadedLocallyAndNotUploadedToMulca() {
        return recordFilter.incompleteAndUploadedLocallyAndNotUploadedToMulca(inProgress()).size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int newestRequestAgeSeconds() {
        return secondsToNow(inProgress()
                .map(MpfsRequestRecord.metaF())
                .map(RequestMeta.createdF)
                .maxO().orElse(startTime));
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int oldestRequestAgeSeconds() {
        return secondsToNow(inProgress()
                .map(MpfsRequestRecord.metaF())
                .map(RequestMeta.createdF)
                .minO().orElse(startTime));
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int oldestRequestUpdateAgeSeconds() {
        return secondsToNow(inProgress()
                .map(MpfsRequestRecord.revisionF())
                .map(RequestRevision.updatedInstantF)
                .minO().orElse(startTime));
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int uploadFromServiceIncompleteCount() {
        return recordFilter.uploadFromServiceIncomplete(inProgress()).size();
    }

    @Safe
    @MonicaMetric
    @GroupByDefault
    public int exportPhotosCountIncompleteCount() {
        return recordFilter.exportPhotosCountIncomplete(inProgress()).size();
    }

    @Safe
    public int uploadFromServiceIncompleteCount(String serviceName) {
        return recordFilter.uploadFromServiceIncomplete(inProgress(), ChemodanService.R.valueOf(serviceName)).size();
    }

    @Safe
    public int exportPhotosCountIncompleteCount(String serviceName) {
        return recordFilter.exportPhotosCountIncomplete(inProgress(), ChemodanService.R.valueOf(serviceName)).size();
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "global-queue",
                new MetricName("queue", "global"),
                "Global queue"
        );
    }

}
