package ru.yandex.chemodan.uploader.registry;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.internal.ReflectionUtils;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.status.MpfsRequestStatus;
import ru.yandex.commune.monitoring.Safe;
import ru.yandex.commune.util.universal.UniversalObjectParserFactory;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.MethodX;

/**
 * @author akirakozov
 */
public class RequestRecordFilter {
    private static final String FILTERS_DESCRIPTION = describeFilters();

    public ListF<MpfsRequestRecord> invokeFilter(String filterName,
            ListF<MpfsRequestRecord> records, ListF<String> params)
    {
        try {
            ListF<MethodX> methods = ClassX.wrap(this.getClass()).getMethods(filterName);
            Check.notEmpty(methods, "No one filter with name " + filterName
                    + " was found, all filters: " + FILTERS_DESCRIPTION);
            Check.isTrue(methods.size() == 1, "More than one filter with name:" + filterName);

            MethodX methodX = methods.get(0);
            Check.equals(methodX.getParameterCount(), params.size() + 1,
                    "Incorrect params count for filter: " + filterName);

            ListF<Object> args = Cf.arrayList();
            args.add(records);
            for (int i = 1; i < methodX.getParameterTypes().size(); ++i) {
                Object o = UniversalObjectParserFactory.consDefault().parse(params.get(i - 1),
                        methodX.getParameterTypes().get(i).getClazz());
                args.add(o);
            }

            return (ListF<MpfsRequestRecord>) methodX.invoke(this, args.toArray());
        } catch (Exception e) {
            throw ReflectionUtils.translate(e);
        }
    }

    public ListF<MpfsRequestRecord> incompleteRequestsWaitingForUser(ListF<MpfsRequestRecord> records) {
        return records.filter(
                MpfsRequestRecord.statusF().andThen(MpfsRequestStatus.isWaitingForExternalF())
            );
    }

    public ListF<MpfsRequestRecord> incompleteAndUploadedToMulca(ListF<MpfsRequestRecord> records) {
        return records.filter(incompleteAndUploadedToMulcaF());
    }


    public ListF<MpfsRequestRecord> incompleteAndUploadedLocally(ListF<MpfsRequestRecord> records) {
        return records.filter(
                MpfsRequestRecord.statusF().andThen(
                        MpfsRequestStatus.isUploadingToDiskF()
                        .andF(MpfsRequestStatus.<MpfsRequestStatus>isLocallyUploadedFileFinishedF())
                ));
    }

    public ListF<MpfsRequestRecord> incompleteAndUploadedLocallyAndNotUploadedToMulca(ListF<MpfsRequestRecord> records)
    {
        return  incompleteAndUploadedLocally(records).filterNot(incompleteAndUploadedToMulcaF());
    }

    @Safe
    public ListF<MpfsRequestRecord> uploadFromServiceIncomplete(
            ListF<MpfsRequestRecord> records, ChemodanService service)
    {
        return records.filter(MpfsRequestRecord.requestF().andThen(isUploadFromServiceF(service)));
    }

    @Safe
    public ListF<MpfsRequestRecord> uploadFromServiceIncomplete(ListF<MpfsRequestRecord> records) {
        return records.filter(MpfsRequestRecord.requestF().andThen(
                (Function1B<MpfsRequest>) r -> r instanceof MpfsRequest.UploadFromService
        ));
    }

    @Safe
    public ListF<MpfsRequestRecord> exportPhotosCountIncomplete(
            ListF<MpfsRequestRecord> records, ChemodanService service)
    {
        return records.filter(MpfsRequestRecord.requestF().andThen(isExportPhotosF(service)));
    }

    @Safe
    public ListF<MpfsRequestRecord> exportPhotosCountIncomplete(ListF<MpfsRequestRecord> records) {
        return records.filter(MpfsRequestRecord.requestF().andThen(
                (Function1B<MpfsRequest>) r -> r instanceof MpfsRequest.ExportPhotos
        ));
    }

    private Function1B<MpfsRequest> isUploadFromServiceF(final ChemodanService service) {
        return new Function1B<MpfsRequest>() {
            public boolean apply(MpfsRequest r) {
                if (r instanceof MpfsRequest.UploadFromService) {
                    return ((MpfsRequest.UploadFromService) r).sourceService == service;
                }
                return false;
            }
        };
    }

    private static Function1B<MpfsRequestRecord> incompleteAndUploadedToMulcaF() {
        return MpfsRequestRecord.statusF().andThen(
                MpfsRequestStatus.isUploadingToDiskF()
                .andF(MpfsRequestStatus.isCommitFileUploadFinishedF())
        );
    }

    private Function1B<MpfsRequest> isExportPhotosF(final ChemodanService service) {
        return new Function1B<MpfsRequest>() {
            public boolean apply(MpfsRequest r) {
                if (r instanceof MpfsRequest.ExportPhotos) {
                    return ((MpfsRequest.ExportPhotos) r).targetService == service;
                }
                return false;
            }
        };
    }

    private static String describeFilters() {
        ListF<String> availableFilters = ClassX.wrap(RequestRecordFilter.class).getDeclaredMethods()
                .filter(MethodX.isPublicF())
                .map(MethodX.nameF())
                .filter(Cf.String.equalsF("invokeFilter").notF());

        return StringUtils.join(availableFilters, ", ");
    }

}
