package ru.yandex.chemodan.uploader.registry;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecordUtils;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.PostProcessStatus;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestDirectorUtils;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.StageListenerPool;
import ru.yandex.commune.uploader.registry.StageProperties;
import ru.yandex.commune.uploader.registry.StageResult;
import ru.yandex.commune.uploader.registry.StageSemaphores;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.StopDueToPermanentFailure;
import ru.yandex.commune.uploader.registry.StopDueToPrematureSuccess;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.DataProgress;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * This class provides api for request processors for
 * updating, controlling and logging any request's state changes
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public class RequestStatesHandler {

    private static final Logger logger = LoggerFactory.getLogger(RequestStatesHandler.class);

    private StageListenerPool<Record<?>> stageListenerPool;
    private StageProperties stageProperties;
    private StageSemaphores stageSemaphores;
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;
    private Stages stages;

    public RequestStatesHandler(
            StageListenerPool<Record<?>> stageListenerPool,
            StageProperties stageProperties, StageSemaphores stageSemaphores,
            UploadRegistry<MpfsRequestRecord> uploadRegistry, Stages stages)
    {
        this.stageListenerPool = stageListenerPool;
        this.stageProperties = stageProperties;
        this.stageSemaphores = stageSemaphores;
        this.uploadRegistry = uploadRegistry;
        this.stages = stages;
    }

    public <R extends MpfsRequestRecord> void withHandleIncomingFileAndFinalize(
            R rawRecord, Function1V<Record<R>> f)
    {
        Record<R> record = Record.cons(uploadRegistry, rawRecord);
        Check.C.isFalse(record.get().getStatus().isFinishedWithFinalStages(),
                "Task " + record.get().meta.id + " has bean already finished.");
        try {
            if (rawRecord.getStatus().waitForExternalField().isPresent()) {
                RequestDirectorUtils.failIfUserDataUploadTimedOut(record, stageProperties);
                RequestDirectorUtils.completeIncomingFileIfNeeded(record);
            }

            if (!rawRecord.getStatus().isFinishedWithoutFinalStages()) { // CHEMODAN-3439
                f.apply(record);
            }
            finalizeRecord(record);
        } catch (StopDueToPermanentFailure | StopDueToPrematureSuccess ex) {
            finalizeRecord(record);
        } catch (RuntimeIoException ex) {
            handleInternalError(record, ex);
            finalizeRecord(record);
        } finally {
            stageListenerPool.getListenerOrNop(record.get().meta.id).apply(record);
        }
    }

    public boolean stageListenerExists(UploadRequestId id) {
        return !stageListenerPool.getListener(id).isPresent();
    }

    public <F> void processSuccess(Record<?> record, MutableState<F> state, Function0<F> op) {
        processSuccess(record, state, op, Option.empty());
    }

    public <F> void process(
            Record<?> record, MutableState<F> state,
            Function<Function1V<DataProgress>, StageResult<F>> op, Option<Duration> timeoutOverride)
    {
        long start = System.nanoTime();
        try {
            RequestDirectorUtils.process(record, state, op, stageProperties, timeoutOverride, stageSemaphores,
                    stageListenerPool.getListenerOrNop(record.get().meta.id));
        } finally {
            logger.debug("Stage {} done in {}ms", record.get().getStatus().fieldName(state).getOrElse("??"),
                    (System.nanoTime() - start) / 1000000L);
        }
    }

    public <F> void processSuccess(
            Record<?> record, MutableState<F> state,
            Function0<F> op, Option<Duration> timeoutOverride)
    {
        process(record, state, stageSuccessWithProgressF(op.asFunction()), timeoutOverride);
    }

    public <F> void process(Record<?> record, MutableState<F> state, Function0<StageResult<F>> op) {
        process(record, state, op.asFunction(), Option.empty());
    }

    public void processCallback(Record<?> record, MutableState<CallbackResponseOption> commitState, String stageName) {
        if (record.get().getRequest().callbackUri.isPresent()) {
            process(record, commitState, stages.commitUploadFileProgressF(record.get(), stageName));
        } else {
            RequestDirectorUtils.skip(record, commitState);
        }
    }

    public void handleInternalError(Record<?> record, Exception ex) {
        record.get().getStatus().internalError.failPermanently(State.initial(), ex);
        record.update(record.get().getStatus().internalError, State.<Boolean>initial().failPermanently(
                new InstantInterval(new Instant(), new Instant()), ex));
        logger.warn(ex, ex);
    }

    public void rollbackOrNot(PostProcessStatus pp, Option<String> yandexCloudRequestId) {
        if (pp.isNeedToRollbackChanges() || pp.commitFileUpload.get().isPermanentFailure()) {
            ListF<MulcaId> uploadedFilesMids = pp.getUploadedFilesMids();
            markMulcaIdsToRemove(uploadedFilesMids, yandexCloudRequestId);
        }
    }

    private static <F> Function<Function1V<DataProgress>, StageResult<F>> stageSuccessWithProgressF(
            final Function<Function1V<DataProgress>, F> op)
    {
        return (callback) -> StageResult.success(op.apply(callback));

    }

    private void finalizeRecord(Record<?> record) {
        if (record.get().getStatus().isFinishedWithoutFinalStages()) {
            MutableState<CallbackResponseOption> commitFinalState = record.get().getStatus().commitFinal;
            Option<PostProcessStatus> postProcessStatusO = record.get().getStatus().getPostProcessStatusO();
            Option<String> yandexCloudRequestId = record.get().getRequest().yandexCloudRequestId;
            if (record.get().getStatus().isNeedToRollbackChanges()) {
                RequestDirectorUtils.skip(record, commitFinalState);
                markMulcaIdsToRemove(postProcessStatusO.get().getUploadedFilesMids(), yandexCloudRequestId);
            } else {
                processCallback(record, commitFinalState, "commitFinal");
                if (postProcessStatusO.isPresent()) {
                    if (commitFinalState.get().isPrematureSuccess()
                            || commitFinalState.get().isPermanentFailure()
                            || postProcessStatusO.get().commitFileUpload.get().isPermanentFailure())
                    {
                        markMulcaIdsToRemove(postProcessStatusO.get().getUploadedFilesMids(), yandexCloudRequestId);
                    }
                }
            }
        }
    }

    private void markMulcaIdsToRemove(ListF<MulcaId> mulcaIds, Option<String> yandexCloudRequestId) {
        if (mulcaIds.isNotEmpty()) {
            MpfsRequest.MarkMulcaIdsForRemove request =
                    new MpfsRequest.MarkMulcaIdsForRemove(ApiVersion.V_0_2, mulcaIds, yandexCloudRequestId);
            uploadRegistry.saveRecord(
                    MpfsRequestRecordUtils.consF(request, RequestRevision.initial(HostInstant.hereAndNow())));
        }
    }
}
