package ru.yandex.chemodan.uploader.registry;

import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.commune.uploader.registry.RecordWrapper;
import ru.yandex.commune.uploader.registry.StageListenerPool;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * StageListenerPool that can be used to add multiple children callbacks - for example, if multiple requests
 * are waiting for the same record that's being processed.
 *
 * @author bursy
 */
public class StageListenerPoolWithChildren<T extends RecordWrapper<?>> extends StageListenerPool<T> {
    private static final Logger logger = LoggerFactory.getLogger(StageListenerPoolWithChildren.class);

    private static final int CHILD_ID_LENGTH = 10;

    private final MapF<UploadRequestId, MapF<String, Consumer<T>>> childListeners = Cf.concurrentHashMap();

    @Override
    public Option<Function1V<T>> getListener(UploadRequestId id) {
        return childListeners.getO(id)
                .map(MapF::values)
                .getOrElse(Cf.list())
                .plus(super.getListener(id))
                .reduceLeftO(Consumer::andThen)
                .map(x -> x::accept);
    }

    public void addChildListener(UploadRequestId id, String childId, Consumer<T> callback) {
        childListeners.getOrElseUpdate(id, Cf::concurrentHashMap).put(childId, callback);
    }

    public void removeChildListener(UploadRequestId id, String childId) {
        childListeners.getO(id).ifPresent(children -> children.removeO(childId));
        childListeners.getO(id).filter(MapF::isEmpty).ifPresent(m -> childListeners.removeO(id));
    }

    public String generateChildId() {
        return Random2.R.nextAlnum(CHILD_ID_LENGTH);
    }
}
