package ru.yandex.chemodan.uploader.registry;


import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.SneakyThrows;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.mime.LibMagicMimeTypeDetector;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.ExtractedFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.av.BaseAntivirus;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.exif.ExifHelper;
import ru.yandex.chemodan.uploader.exif.ExifTool;
import ru.yandex.chemodan.uploader.installer.InstallerModifier;
import ru.yandex.chemodan.uploader.installer.InstallerModifierParams;
import ru.yandex.chemodan.uploader.log.OauthCodeResult;
import ru.yandex.chemodan.uploader.mpfs.MpfsUtils;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.preview.AlbumPreviewManager;
import ru.yandex.chemodan.uploader.preview.PreviewImageManager;
import ru.yandex.chemodan.uploader.preview.PreviewInfo;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.record.status.DigestCalculationStatus;
import ru.yandex.chemodan.uploader.registry.record.status.DownloadedInMemoryMulcaFile;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.registry.record.status.MediaInfo;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewVideoStatus;
import ru.yandex.chemodan.uploader.registry.record.status.UploadedMulcaFile;
import ru.yandex.chemodan.uploader.services.ServiceApis;
import ru.yandex.chemodan.uploader.services.ServiceFileId;
import ru.yandex.chemodan.uploader.services.ServiceFileInfo;
import ru.yandex.chemodan.uploader.services.ServiceImageInfo;
import ru.yandex.chemodan.uploader.services.ServiceIncomingFile;
import ru.yandex.chemodan.uploader.services.ServiceIncomingImage;
import ru.yandex.chemodan.uploader.social.ExternalResourceClient;
import ru.yandex.chemodan.uploader.social.SocialDataFormatUtils;
import ru.yandex.chemodan.util.exception.PermanentFailureException;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.chemodan.util.http.HttpClientConfigurator;
import ru.yandex.chemodan.util.oauth.OauthClient;
import ru.yandex.commune.archive.ArchiveEntry;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.archive.ArchiveManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.image.imageMagick.ImageMagick;
import ru.yandex.commune.image.imageMagick.ImageMagickMontage;
import ru.yandex.commune.image.imageMagick.MontageTile;
import ru.yandex.commune.json.JsonArray;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.serialize.JsonParser;
import ru.yandex.commune.json.serialize.JsonParserException;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.RequestRecord;
import ru.yandex.commune.uploader.registry.StageResult;
import ru.yandex.commune.uploader.registry.StageUtils;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.http.Content;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.ContentUtils;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.commune.uploader.util.http.MultipartUtils;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.util.http.UploadedPartInfo;
import ru.yandex.commune.uploader.web.data.DiskWritePolicy;
import ru.yandex.commune.uploader.web.data.ProcessPutException;
import ru.yandex.commune.video.FfTools;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaException;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderJsonParser;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.IoFunction;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.RuntimeIoException;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.os.CpuBound;
import ru.yandex.misc.time.Stopwatch;
import ru.yandex.misc.webdav.DeltaEncoding;
import ru.yandex.misc.webdav.WebDavDigest;

/**
 * @author vavinov
 */
public class Stages {
    private static final Logger logger = LoggerFactory.getLogger(Stages.class);

    @Autowired
    private ArchiveManager archiveManager;
    @Autowired
    private MulcaClient mulcaClient;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    private DocviewerClient docviewerClient;
    @Autowired
    private BaseAntivirus antivirus;
    @Autowired
    private CpuBound cpuBound;
    @Autowired
    private PreviewImageManager previewImageManager;
    @Autowired
    private AlbumPreviewManager albumPreviewManager;
    @Autowired
    private FfTools ffTools;
    @Autowired
    private MpfsClient mpfsClient;
    @Autowired
    @Qualifier("mpfs_callback")
    private HttpClient mpfsCallbackHttpClient;
    @Autowired
    private DiskWritePolicy diskWritePolicy;
    @Autowired
    @Qualifier("externalResourceClient")
    private ExternalResourceClient externalResourceClient;
    @Autowired
    private ServiceApis serviceApis;
    @Autowired
    private OauthClient oauthClient;
    @Autowired
    @Qualifier("avatars")
    private HttpClientConfigurator avatarsHttpClientConfigurator;
    @Autowired
    private LibMagicMimeTypeDetector libMagicMimeTypeDetector;

    @Value("${uploader.mulca.identifier}")
    private String uploaderMulcaIdentifierPrefix;
    @Value("${uploader.mulca.identifier.append.uid}")
    private boolean uploaderMulcaIdentifierAppendUid;
    @Value("${uploader.mulca.upload.enabled:-true}")
    private boolean mulcaUploadEnabled;
    @Value("${archive.max.size}")
    private DataSize maxArchiveSize;
    @Value("${archive.max.entries}")
    private int maxArchiveEntries;
    @Value("${extract.exif.enabled}")
    private boolean extractExifEnabled;
    @Value("${extract.videoinfo.enabled}")
    private boolean extractVideoInfoEnabled;
    @Value("${uploader.store.bad.patch}")
    private boolean needToStoreBadPatch;
    @Value("${uploader.bad.patches.dir}")
    private String badPatchesDir;
    @Value("${preview.video.enabled}")
    private boolean previewVideoEnabled;
    @Value("${preview.video.multiple.enabled}")
    private boolean multiplePreviewVideoEnabled;
    @Value("${preview.image.imagemagick.ping.timeout}")
    private Duration imageMagickPingTimeout;
    @Value("${oauth.code.ttl}")
    private Duration oauthCodeTtl;
    @Value("${mulca.http.socket.timeout}")
    private Duration mulcaSocketTimeout;
    @Value("${mulca.upload.socket.timeout.coefficient}")
    private double mulcaUploadSocketTimeoutCoefficient;
    @Value("${avatars.upload.host}")
    private String avatarsUploadHost;
    @Value("${avatars.upload.namespace}")
    private String avatarsUploadNamespace;
    @Value("${docviewer.enabled}")
    private boolean docViewerEnabled;

    private final BenderJsonParser<AvatarsMeta> avatarsMetaParser = Bender.jsonParser(AvatarsMeta.class);
    private final BenderJsonParser<AvatarsUploadResponse> avatarsUploadResponseParser = Bender.jsonParser(AvatarsUploadResponse.class);
    private final Pattern avatarsStidPattern = Pattern.compile("^ava:([^:]+):([^:]+):([^:]+)$");
    private final DynamicProperty<Boolean> logAvatarsUploadResponse = new DynamicProperty<>("avatars-log-upload-response", false);
    private final DynamicProperty<Long> avatarsUploadTimeout = new DynamicProperty<>("avatars-upload-timeout", 60_000L);
    private final DynamicProperty<Boolean> compareMimeTypeWithLibMagic =
            new DynamicProperty<>("uploader-compare-mimetype-with-libmagic", false);
    private final DynamicProperty<Boolean> correctMimeTypeWithLibMagic =
            new DynamicProperty<>("uploader-correct-mimetype-with-libmagic", false);

    private static Option<String> getBody(HttpResponse response) {
        return Option.ofNullable(response.getEntity()).map(e -> {
            try {
                return InputStreamSourceUtils.wrap(e.getContent()).readText("utf-8");
            } catch (IOException e1) {
                throw new RuntimeException(e1);
            }
        });
    }

    MulcaId handleAvatarsResponse(HttpResponse r, AtomicReference<AvatarsMeta> meta) {
        String body = getBody(r).getOrNull();
        if (logAvatarsUploadResponse.get()) {
            logger.info(body);
        }
        if (HttpStatus.is2xx(r.getStatusLine().getStatusCode())) {
            AvatarsUploadResponse ar = avatarsUploadResponseParser.parseJson(body);
            meta.set(ar.getMeta());
            return MulcaId.fromSerializedString(String.format(
                    "ava:%s:%s:%s",
                    avatarsUploadNamespace,
                    ar.getGroupId(),
                    ar.getImageName()
            ));
        } else if(r.getStatusLine().getStatusCode() == HttpStatus.SC_415_UNSUPPORTED_MEDIA_TYPE) {
            throw new MediaTypeNotSupportedException("Unsupported avatar media type exception code: " + r.getStatusLine().getStatusCode() +
                    "body: " + r.getStatusLine() + "\n" + body);
        } else {
            try {
                JsonParser
                        .getInstance()
                        .parseObject(body)
                        .getByPathO("description")
                        .cast(JsonString.class)
                        .map(JsonString::getString)
                        .ifPresent(d -> {
                            throw new AvatarsException(r.getStatusLine().getStatusCode() + " " + d);
                        });
            } catch (JsonParserException e) {
                // fallback to HttpException
            }
            throw new HttpException(r.getStatusLine().getStatusCode(), r.getStatusLine() + "\n" + body);
        }
    }

    public MulcaUploadInfo uploadToMulca(UidOrSpecial uidOrSpecial, InputStreamSource data) {
        if (mulcaUploadEnabled) {
            String identifier = uploaderMulcaIdentifierPrefix;
            if (uploaderMulcaIdentifierAppendUid) identifier += uidOrSpecial.toSpecialOrUidString();

            try {
                Option<RequestConfig> config = getRequestConfigForMulcaUpload(data);
                return new MulcaUploadInfo(mulcaClient.upload(data, identifier, config, false, true), data.lengthO());
            } catch (MulcaException ex) {
                if (HttpStatus.is4xx(ex.getStatusCode())) {
                    throw new PermanentFailureException(ex.getMessage(), ex);
                } else {
                    logger.warn("Failed to upload data to storage: {}, code {}",
                            ex.getMessage(), ex.getStatusCode());
                    throw ex;
                }
            }
        } else {
            return MulcaUploadInfo.fromMulcaId(MulcaId.fromSerializedString("dummy.mulca.id"));
        }
    }

    private Option<RequestConfig> getRequestConfigForMulcaUpload(InputStreamSource data) {
        return data.lengthO().map(length -> {
            long timeout = calculateDynamicSocketTimeoutForMulcaUpload(length);
            logger.info("Dynamic socket timeout for mulca upload in ms: " + timeout);
            return RequestConfig.custom().setSocketTimeout((int) timeout).build();
        });
    }

    @SneakyThrows
    public Option<AvatarsMeta> getAvatarsMeta(MulcaId mulcaId) {
        Matcher m = avatarsStidPattern.matcher(mulcaId.getStid());
        if (m.matches()) {
            String url = String.format("http://%s/getinfo-%s/%s/%s/meta", avatarsUploadHost, m.group(1), m.group(2), m.group(3));
            return avatarsHttpClientConfigurator.configure().execute(new HttpGet(url), r -> {
                Option<String> body = getBody(r);
                if (HttpStatus.is2xx(r.getStatusLine().getStatusCode())) {
                    return body.map(avatarsMetaParser::parseJson);
                } else {
                    throw new HttpException(r.getStatusLine().getStatusCode(), body.toString());
                }
            });
        } else {
            return Option.empty();
        }
    }

    private long calculateDynamicSocketTimeoutForMulcaUpload(long size) {
        DataSize dataSize = DataSize.fromBytes(size);
        long dynamicSocketTimeout = (long) (dataSize.toMegaBytes() * mulcaUploadSocketTimeoutCoefficient * 1000);
        return Math.max(mulcaSocketTimeout.getMillis(), dynamicSocketTimeout);
    }

    private MulcaUploadInfo uploadFileToAvatars(Supplier<HttpUriRequest> supplier, AtomicReference<AvatarsMeta> meta) {
        try {
            long start = System.currentTimeMillis();
            logger.info("Uploading file to avatars...");
            MulcaId stid = avatarsHttpClientConfigurator
                    .withCustomTimeout(Timeout.milliseconds(avatarsUploadTimeout.get()))
                    .execute(supplier.get(), r -> handleAvatarsResponse(r, meta));
            logger.info("Stored file in avatars, id={}, time={}ms", stid, System.currentTimeMillis() - start);
            return MulcaUploadInfo.fromMulcaId(stid);
        } catch (Exception e) {
            logger.warn("Failed to upload file to avatars", e);
            throw ExceptionUtils.translate(e);
        }
    }

    private String avatarsUploadUrl() {
        return String.format("http://%s/put-%s", avatarsUploadHost, avatarsUploadNamespace);
    }

    public Function0<MulcaUploadInfo> uploadFileToAvatarsF(MulcaId stid, AtomicReference<AvatarsMeta> meta) {
        return () -> uploadFileToAvatars(() -> new HttpGet(String.format("%s?stid=%s", avatarsUploadUrl(), stid.getStid())), meta);
    }

    public Function0<MulcaUploadInfo> uploadFileToAvatarsF(InputStreamSource localFile, AtomicReference<AvatarsMeta> meta) {
        return () -> uploadFileToAvatars(() -> new HttpPost(avatarsUploadUrl()) {{
            setEntity(MultipartEntityBuilder.create().addBinaryBody("file", localFile.getInputUnchecked()).build());
        }}, meta);
    }

    public Function0<MulcaUploadInfo> uploadFileToMulcaF(
            final InputStreamSource localFile, final UidOrSpecial uid)
    {
        return () -> {
            logger.info("Uploading file to mulca...");
            MulcaUploadInfo uploadInfo = uploadToMulca(uid, localFile);

            logger.info("Stored file in mulca, id={}", uploadInfo.getMulcaId());
            return uploadInfo;
        };
    }

    public Function0<MulcaUploadInfo> uploadStringToMulcaF(final String data, final UidOrSpecial uid) {
        return uploadFileToMulcaF(
                new ByteArrayInputStreamSource(data.getBytes(CharsetUtils.UTF8_CHARSET)), uid);
    }

    public Function0<DigestCalculationStatus> calculateDigestF(
            final UploadRequestId requestId,
            final InputStreamSource localFile)
    {
        return () -> {
            logger.info("Calculating digest...");
            final File2 digestFile = localFileManager.allocateFile(requestId, "digest");
            cpuBound.withSemaphore(() -> {
                WebDavDigest.digest(localFile, digestFile.asOutputStreamTool());
                return null;
            });
            return new DigestCalculationStatus(digestFile);
        };
    }

    public Function0<MulcaUploadInfo> uploadDigestToMulcaF(
            final File2 digestFile,
            final UidOrSpecial uidOrSpecial)
    {
        return () -> {
            logger.info("Uploading file digest to mulca...");
            MulcaUploadInfo uploadInfo = uploadToMulca(uidOrSpecial, digestFile);
            logger.info("Stored digest in mulca, id={}", uploadInfo.getMulcaId());
            return uploadInfo;
        };
    }

    public Function0<CallbackResponseOption> markMulcaIdToRemoveF(final MulcaId mulcaId) {
        return () -> CallbackResponseOption.some(MpfsUtils.convert(mpfsClient.markMulcaIdToRemove(mulcaId)));
    }

    public Function0<StageResult<File2>> generatePreviewForDocumentF(
            final UploadRequestId requestId, final MulcaId fileMulcaId,
            final UidOrSpecial uid, final String contentType)
    {
        return () -> {
            if (!docViewerEnabled) {
                return StageResult.disabled();
            }

            File2 preview = File2.withNewTempDir(tempDir -> {
                File2 downloaded = tempDir.child("downloaded");
                try {
                    docviewerClient.getDocumentPreview(fileMulcaId, uid.toPassportUidOrAnonymous(), contentType)
                            .readTo(downloaded);

                } catch (HttpException e) {
                    if (e.getStatusCode().isPresent() && HttpStatus.is4xx(e.getStatusCode().get())) {
                        // Special exception, to avoid retry for 4xx error codes
                        throw new GeneratePreviewException(e);
                    } else {
                        throw e;
                    }
                }

                File2 result = localFileManager.allocateFile(requestId, "dv-preview");
                downloaded.renameTo(result);

                return result;
            });

            return StageResult.success(preview);
        };
    }

    public Function0<StageResult<File2>> generatePreviewForVideoF(
            final UploadRequestId requestId, final Either<InputStreamSource, MulcaId> source,
            final Duration videoDuration, final Option<Dimension> dimension)
    {
        return () -> {
            if (!previewVideoEnabled) {
                return StageResult.disabled();
            }

            File2 directory = localFileManager.allocateFile(requestId, "video-previews");
            directory.mkdirs();
            File2 resultPreview = directory.child(PreviewVideoStatus.getPreviewFileName());

            Duration firstScreenshotOffset = videoDuration.isLongerThan(Duration.standardSeconds(2)) ?
                    Duration.standardSeconds(1) : Duration.ZERO;

            RuntimeException lastException = null;
            for (Duration duration : Cf.list(new Duration(videoDuration.getMillis() / 3), firstScreenshotOffset)) {
                try {
                    logger.info("Generation preview for time {} -> {}", duration, resultPreview);
                    Stopwatch stopwatch = Stopwatch.createAndStart();

                    InputStreamSource actualSource =
                            source.isLeft() ? source.getLeft() : mulcaClient.download(source.getRight());
                    File2 file = localFileManager.inStreamSourceToFile2(requestId, actualSource, "video-source");
                    ffTools.takeScreenshot(file, duration, resultPreview, Option.empty(), FfTools.ScreenshotMode.FAST);

                    stopwatch.stop();
                    logger.info("preview generated in {}", stopwatch.millisDuration());
                    break;
                } catch (RuntimeException e) {
                    lastException = e;
                    logger.warn("Failed to generate screenshot {}: {}", duration, e.getMessage());
                }
            }
            if (lastException != null) {
                throw lastException;
            }

            if (multiplePreviewVideoEnabled && dimension.isPresent()) {
                ListF<Duration> previewTimes = PreviewVideoStatus.getMultiplePreviewTimes(videoDuration);
                ListF<File2> generated = Cf.arrayList();
                for (Tuple2<Duration, Integer> tuple2 : previewTimes.zipWithIndex()) {
                    Duration duration = tuple2._1;
                    File2 target = directory.child(PreviewVideoStatus.getMultiplePreviewFileName(tuple2._2));

                    logger.info("Generation preview for time {} -> {}", duration, target);
                    try {
                        InputStreamSource actualSource =
                                source.isLeft() ? source.getLeft() : mulcaClient.download(source.getRight());
                        File2 file = localFileManager.inStreamSourceToFile2(requestId, actualSource, "video-source");
                        ffTools.takeScreenshot(file, duration, resultPreview, Option.empty(),
                                FfTools.ScreenshotMode.FAST);

                        generated.add(target);
                    } catch (RuntimeException e) {
                        logger.warn("Failed to generate screenshot {}: {}", duration, e.getMessage());
                        target.deleteRecursiveQuietly();
                    }
                }

                if (generated.isNotEmpty()) {
                    try {
                        //TODO: path to config
                        Dimension previewDimension = dimension.get();
                        new ImageMagickMontage("/usr/bin/montage")
                                .montage(previewDimension,
                                        Option.of(new MontageTile(generated.size(), 1)),
                                        directory.child(PreviewVideoStatus.getMultipleMontagedPreviewFileName()),
                                        generated.toArray(File2.class));
                    } catch (RuntimeException e) {
                        logger.warn("Failed to montage multiple previews");
                    }
                }
            }

            return StageResult.success(directory);
        };
    }

    /**
     * @param serviceImageInfo if defined properties from service file will be used in ExifInfo
     */
    public Function0<StageResult<ExifInfo>> extractExifInfoF(final UploadRequestId requestId,
            final InputStreamSource source, final Option<ServiceImageInfo> serviceImageInfo,
            final String contentType, final Function1V<File2> logExifF)
    {
        return () -> {
            if (!extractExifEnabled) {
                return StageResult.disabled();
            }
            File2 src = localFileManager.inStreamSourceToFile2(requestId, source, "for-exif");

            logExifF.apply(src);

            return StageResult.success(ExifHelper.extractExif(src, contentType, serviceImageInfo));
        };
    }

    public Function0<Option<ExifInfo>> tryExtractExifF(File2 downloaded, String contentType) {
        return () -> {
            try {
                ExifInfo exif = ExifHelper.extractExif(downloaded, contentType, Option.empty());
                return Option.when(exif.isPresent(), exif);
            } catch (Exception e) {
                logger.warn(e);
                return Option.empty();
            }
        };
    }

    public Function0<String> extractFullExifInJsonF(final File2 source) {
        return new Function0<String>() {
            @Override
            public String apply() {
                String exif = ExifTool.INSTANCE.getFullExifInJson(source);
                return containsRealExifInfo(exif) ? exif : "";
            }

            private boolean containsRealExifInfo(String exif) {
                JsonArray arr = (JsonArray) JsonParser.getInstance().parse(exif);
                JsonObject dict = (JsonObject) arr.getArray().get(0);
                if (dict.size() == 1 && dict.getO("SourceFile").isPresent()) {
                    logger.debug("No exif info found");
                    return false;
                }
                return true;
            }
        };
    }

    public Function0<StageResult<GenerateImageOnePreviewResult>> generateOnePreviewForImageF(
            UploadRequestId requestId, InputStreamSource source, PreviewSizeParameter size, boolean onFly,
            boolean crop, final Option<Integer> quality)
    {
        return generateOnePreviewForImageF(requestId, source, Option.empty(), size, onFly, crop, quality);
    }

    public Function0<StageResult<GenerateImageOnePreviewResult>> generateOnePreviewForImageF(
            final UploadRequestId requestId, final InputStreamSource source, final Option<String> contentType,
            final PreviewSizeParameter size, final boolean onFly, final boolean crop, final Option<Integer> quality) {
        return () -> {
            if (!onFly && !previewImageManager.isEnabledForOne()) {
                return StageResult.disabled();
            }

            File2 original = localFileManager.inStreamSourceToFile2(requestId, source, "preview-for-image");
            File2 previewFile = original.sibling("preview");
            PreviewInfo previewInfo = previewImageManager.generateOnePreviewForImage(
                    original, contentType, size.asDimension(), crop, previewFile.getName(), quality);

            // https://jira.yandex-team.ru/browse/CHEMODAN-17145
            if (previewFile.getFile().length() == 0) {
                logger.warn("Preview file is empty");
                return StageResult.tempFailure();
            }
            return StageResult.success(new GenerateImageOnePreviewResult(previewFile, previewInfo));
        };
    }

    public Function0<PreviewInfo> generateAlbumPreviewF(final File2 source, final String albumName, final String userName) {
        return () -> albumPreviewManager.generateAlbumPreview(source, albumName, userName);
    }

    public Function0<PreviewInfo> generateOnePreviewForImageF(final File2 source, final String previewFileName,
            final PreviewSizeParameter size, final boolean crop, final Option<Integer> quality)
    {
        return () -> previewImageManager.generateOnePreviewForImage(
                source, Option.empty(), size.asDimension(), crop, previewFileName, quality);
    }

    public Function0<StageResult<AntivirusResult>> checkWithAntivirusF(
            Option<UploadRequestId> reqId, final InputStreamSource source)
    {
        return () -> {
            if (!antivirus.isEnabled()) {
                return StageResult.disabled();
            }

            logger.info("Checking file with antivirus...");

            // XXX: upyachka for fast migration to new antivirus
            // 1. copy one more time files from browsers
            // 2. nonconsistent properties: stdinOnly vs fileOnly
            // 3. not nice solution with Option for reqId
            InputStreamSource newSource = source;
            if (reqId.isPresent() && antivirus.supportFileOnly()) {
                newSource = localFileManager.inStreamSourceToFile2(reqId.get(), source, "antivirus");
            }
            return StageResult.success(antivirus.check(newSource));
        };
    }

    public Function0<UploadedMulcaFile> downloadFromMulcaF(MulcaId mulcaId, File2 destinationFile) {
        return () -> {
            try {
                return new UploadedMulcaFile(downloadToLocalFileF(mulcaClient.download(mulcaId), destinationFile, false, false).apply(), mulcaId);
            } catch (RuntimeIoException e) {
                throw new MulcaDownloadException(e, mulcaId);
            }
        };
    }

    public Function0<DownloadedInMemoryMulcaFile> downloadFromMulcaToMemoryF(final MulcaId mulcaId) {
        return () ->  {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            try {
                mulcaClient.download(mulcaId).readTo(out);
            } catch (HttpException ex) {
                int statusCode = ex.getStatusCode().orElse(0);
                if (HttpStatus.is4xx(statusCode)) {
                    throw new PermanentHttpFailureException(ex.getMessage(), ex);
                } else {
                    logger.warn("Failed to download file: {}, code {}",
                            ex.getMessage(), ex.getStatusCode());
                    throw ex;
                }
            }

            return new DownloadedInMemoryMulcaFile(out.toByteArray(), mulcaId);
        };
    }

    public Function0<UploadedMulcaFile> downloadFromMulcaF(final UploadRequestId requestId, final MulcaId mulcaId) {
        return downloadFromMulcaF(mulcaId, localFileManager.allocateFile(requestId, "download"));
    }

    private Function0<File2> downloadToLocalFileF(final InputStreamSource source, final File2 result,
            final boolean disableRetries, final boolean disableRedirects)
    {
        return () -> File2.withNewTempDir(tempDir -> {
            File2 downloaded = tempDir.child("downloaded");
            try {
                source.readTo(downloaded);
            } catch (HttpException ex) {
                int statusCode = ex.getStatusCode().orElse(0);
                if (disableRetries || HttpStatus.is4xx(statusCode)) {
                    throw new PermanentHttpFailureException(ex.getMessage(), ex);
                } else if (disableRedirects &&
                        (HttpStatus.SC_301_MOVED_PERMANENTLY == statusCode ||
                        HttpStatus.SC_302_MOVED_TEMPORARILY == statusCode))
                {
                    throw new PermanentFailureException(ex.getMessage(), ex);
                } else {
                    logger.warn("Failed to download file: {}, code {}",
                            ex.getMessage(), ex.getStatusCode());
                    throw ex;
                }
            } catch (Exception e) {
                if (disableRetries) {
                    throw new PermanentFailureException(e.getMessage(), e);
                }
                logger.warn("Failed to download file", e);
                throw e;
            }

            downloaded.renameTo(result);

            return result;
        });
    }

    public Function0<File2> patchLocalFileF(
            final UploadRequestId requestId,
            final File2 original, final String expectedOriginalMd5,
            final InputStreamSource patch, final String expectedResultMd5,
            final Option<DataSize> maxFileSizeO)
    {
        return () -> {
            Check.C.equals(expectedOriginalMd5, Md5.A.digest(original).hex(),
                    "Original file md5 mismatch");
            return File2.withNewTempDir(tempDir -> {
                File2 patched = tempDir.child("patched");
                DeltaEncoding.applyPatch(patch, original, patched);
                if (ObjectUtils.equals(expectedResultMd5, Md5.A.digest(patched).hex())) {
                    validateFileSize(DataSize.fromBytes(patched.length()));
                    long additionalSize = patched.length() - original.length();
                    if (additionalSize > 0) {
                        validateFreeDiskSpace(DataSize.fromBytes(additionalSize), maxFileSizeO);
                    }

                    File2 result = localFileManager.allocateFile(requestId, "patch");
                    patched.renameTo(result);
                    return result;
                } else {
                    if (needToStoreBadPatch) {
                        try {
                            File2 dir = new File2(badPatchesDir);
                            dir.mkdirs();
                            File2 badPatchFile = dir.child(requestId.id);
                            logger.debug("Store bad patch in file: " + badPatchFile.getPath());
                            IoUtils.copy(patch.getInput(), badPatchFile.asOutputStreamTool().getOutput());
                        } catch (Exception e) {
                            logger.warn("Couldn't save bad patch: " + e.getMessage());
                        }
                    }
                    throw new IllegalStateException("Patched file md5 mismatch");
                }
            });
        };
    }

    public Function0<ContentInfo> contentInfoF(Content content, File2 file, Option<String> filename) {
        return contentInfoF(content, () -> detectMimeType(content, file, filename));
    }

    public Function0<ContentInfo> contentInfoF(IncomingFile incomingFile, Option<String> filename) {
        Content content = ContentUtils.multipartOrRaw(incomingFile);

        return contentInfoF(content, MultipartUtils.isMultipartFormData(incomingFile)
                                     ? () -> ContentUtils.getContentType(content, filename)
                                     : () -> detectMimeType(content, incomingFile.getRawFile(), filename));
    }

    private Function0<ContentInfo> contentInfoF(Content content, Supplier<Option<String>> detectMimeTypeF) {
        return () -> cpuBound.withSemaphore(() ->
                ContentUtils.info(content.getInputStreamSource(), detectMimeTypeF.get()));
    }

    public Option<String> detectMimeType(Content content, File2 file, Option<String> filename) {
        Option<String> contentType = ContentUtils.getContentType(content, filename);

        if (compareMimeTypeWithLibMagic.get()) {
            Option<String> libMagicMimeType = detectMimeTypeByLibMagicAndCompare(contentType, file, filename);

            if (correctMimeTypeWithLibMagic.get()) {
                return libMagicMimeType;
            }
        }

        return contentType;
    }

    private Option<String> detectMimeTypeByLibMagicAndCompare(Option<String> detectedMimeType, File2 file,
            Option<String> filename)
    {
        try {
            Stopwatch sw = Stopwatch.createAndStart();
            Option<String> libMagicMimeType = libMagicMimeTypeDetector.detectWithOverrides(file, filename);
            sw.stop();

            logger.info("mime type comparison: MimeTypeDetector detected={}, libmagic detected={}, "
                            + "types are equal={}, time taken by libmagic={}",
                    detectedMimeType, libMagicMimeType, detectedMimeType.equals(libMagicMimeType),
                    sw.millisDuration());

            return libMagicMimeType;
        } catch (Exception e) {
            logger.warn("libmagic mime type detection failed", e);
        }

        return Option.empty();
    }

    public Function0<ArchiveListing> listArchiveF(final File2 archiveFile) {
        return () -> {
            checkArchiveSize(archiveFile);
            ArchiveListing listArchive = archiveManager.listArchive(archiveFile);
            if (listArchive.getEntries().length() > maxArchiveEntries) {
                throw new PermanentFailureException("Archive too big: " + listArchive.getEntries().length() + " entries", null);
            }
            return listArchive;
        };
    }

    public Function0<IncomingFile> extractFileFromArchiveF(final UploadRequestId id,
            final File2 archiveFile, final String fileToExtract, Option<DataSize> maxFileSizeO)
    {
        return () -> {
            ListF<ExtractedFile> extractedFiles = extractArchiveF(id, archiveFile, Option.of(fileToExtract),
                    maxFileSizeO).apply();
            Option<ExtractedFile> extractedFileO = extractedFiles.firstO();
            if (extractedFileO.isPresent()) {
                File2 localFile = extractedFileO.get().getLocalFile();
                return new IncomingFile(Option.empty(),
                        Option.of(DataSize.fromBytes(localFile.length())), localFile);
            } else {
                throw new PermanentFailureException(String.format("'%s' is a directory, not a file", fileToExtract));
            }
        };
    }

    public Function0<IncomingFile> convertFileToMsOoxmlFormatF(final UploadRequestId requestId,
            URI url, final UidOrSpecial uid, final Option<String> archivePath, final Option<String> contentType)
    {
        return () -> {
            if (!docViewerEnabled) {
                throw new PermanentFailureException("Can't convert file " + url + " for uid" +
                        uid.toPassportUidOrAnonymous().getUid() + ". Docviewer client is disabled");
            }

            File2 convertedFile = File2.withNewTempDir(tempDir -> {
                File2 downloaded = tempDir.child("downloaded");
                try {
                    docviewerClient
                            .convertToMsOoxmlFormat(url, uid.toPassportUidOrAnonymous(), archivePath, contentType)
                            .readTo(downloaded);
                } catch (HttpException e) {
                    if (e.getStatusCode().isPresent() && HttpStatus.is4xx(e.getStatusCode().get())) {
                        // Special exception, to avoid retry for 4xx error codes
                        throw new ConvertToMsOoxmlFormatException(e);
                    } else {
                        throw e;
                    }
                }

                File2 result = localFileManager.allocateFile(requestId, "dv-converted-to-ms-ooxml");
                downloaded.renameTo(result);

                return result;
            });

            return new IncomingFile(Option.empty(),
                    Option.of(DataSize.fromBytes(convertedFile.length())), convertedFile);
        };
    }

    public Function0<ListF<ExtractedFile>> extractArchiveF(final UploadRequestId id, final File2 archiveFile,
            final Option<String> fileToExtract, Option<DataSize> maxFileSizeO)
    {
        return () -> {
            checkArchiveSize(archiveFile);
            ArchiveAcceptor aa = new ArchiveAcceptor(id);
            if (fileToExtract.isPresent()) {
                archiveManager.extractOne(archiveFile, fileToExtract.get(), aa.consumer());
            } else {
                archiveManager.extractAll(archiveFile, aa.consumer());
            }
            aa.closeStreams();
            validateExtractedFileSizes(aa.extractedFiles, maxFileSizeO);
            return aa.extractedFiles;
        };
    }

    private void validateExtractedFileSizes(ListF<ExtractedFile> extractedFiles, Option<DataSize> maxFileSizeO) {
        DataSize sumSize = DataSize.fromBytes(0);
        for (ExtractedFile extractedFile : extractedFiles) {
            DataSize fileSize = DataSize.fromBytes(extractedFile.getLocalFile().length());
            validateFileSize(fileSize);
            sumSize.plus(fileSize);
        }
        validateFreeDiskSpace(sumSize, maxFileSizeO);
    }

    private void checkArchiveSize(final File2 archiveFile) {
        if (archiveFile.length() > maxArchiveSize.toBytes()) {
            throw new PermanentFailureException("Archive too big: " + archiveFile.length() + " bytes", null);
        }
    }

    public Function0<Integer> extractRotationF(File2 downloaded) {
        return () -> ExifTool.getRotateAngle(downloaded, new ImageMagick());
    }

    private class ArchiveAcceptor {
        ListF<ExtractedFile> extractedFiles = Cf.arrayList();
        ListF<OutputStream> extractedFilesOutputStreams = Cf.arrayList();
        UploadRequestId id;

        ArchiveAcceptor(UploadRequestId id) {
            this.id = id;
        }

        Function<ArchiveEntry, OutputStream> consumer() {
            return (IoFunction<ArchiveEntry, OutputStream>) entry -> {
                File2 file = localFileManager.allocateFile(id, "extracted");
                if (!entry.isFolder()) {
                    extractedFiles.add(new ExtractedFile(entry, file));
                }
                OutputStream out = new BufferedOutputStream(new FileOutputStream(file.getFile()));
                extractedFilesOutputStreams.add(out);
                return out;
            };
        }

        void closeStreams() {
            extractedFilesOutputStreams.forEach(IoUtils::closeQuietly);
        }
    }

    private static void validateFreeDiskSpace(DataSize additionalSize, Option<DataSize> maxFileSizeO) {
        maxFileSizeO.filter(additionalSize::gt).ifPresent(maxFileSize -> {
            throw new ProcessPutException(PutResult.INSUFFICIENT_DISK_SPACE,
                    "Insufficient disk space, additional size=" + additionalSize +
                            ", max-file-size=" + maxFileSize);
        });
    }

    private void validateFileSize(DataSize fileSize) {
        if (fileSize.gt(diskWritePolicy.getUploadLengthLimit())) {
            throw new PermanentFailureException(
                    "File is too large: " + fileSize + ", limit=" + diskWritePolicy.getUploadLengthLimit());
        }
    }

    public Function0<StageResult<MediaInfo>> extractMediaInfoF(Option<Instant> creationTs) {
        // It's legacy part, by historical reason we extracted media info before video info for video files,
        // now we have creation time from ffprobe output, so we don't need te reextract it.
        // But mpfs looks at this stage, so we should save this stage util mpfs would take all from videoInfo result.
        return () -> StageResult.success(new MediaInfo(creationTs));
    }

    public Function0<StageResult<FileInformation>> extractVideoInfoF(
            final UploadRequestId requestId, final InputStreamSource source)
    {
        return () -> {
            if (!extractVideoInfoEnabled) {
                return StageResult.disabled();
            }

            File2 src = localFileManager.inStreamSourceToFile2(requestId, source, "for-video-info");
            FileInformation result = ffTools.videoInformation(src);
            Check.isTrue(result.hasVideoStreams(), "No video streams");
            return StageResult.success(result);
        };
    }

    public Function0<StageResult<FileInformation>> extractVideoInfoF(final UploadRequestId requestId,
            final MulcaId mulcaId)
    {
        return () -> {
            if (!extractVideoInfoEnabled) {
                return StageResult.disabled();
            }

            File2 originalFile = downloadFromMulcaF(requestId, mulcaId).apply().getLocalFile();
            FileInformation result = ffTools.videoInformation(originalFile);
            Check.isTrue(result.hasVideoStreams(), "No video streams");
            return StageResult.success(result);
        };
    }

    Function0<StageResult<CallbackResponseOption>> commitUploadFileProgressF(
            RequestRecord record, String stageName)
    {
        return StageUtils.commitUploadFileProgressF(record, Option.of(stageName), Option.of(mpfsCallbackHttpClient),
                Cf.set(HttpStatus.SC_201_CREATED, HttpStatus.SC_409_CONFLICT));
    }

    public Function0<ServiceIncomingImage> downloadFileFromServiceF(
            final UploadRequestId id,
            final ChemodanService sourceService,
            final Option<Boolean> disableRetriesO,
            final Option<Boolean> disableRedirectsO,
            final Option<ServiceImageInfo> serviceImageInfoO,
            final Option<ServiceFileId> serviceFileIdO,
            final Option<MulcaId> aviaryOriginalMulcaIdO,
            final Option<DataSize> maxFileSizeO,
            final Option<Boolean> excludeOrientationO)
    {
        return () -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();

            Content content = getContent(sourceService, serviceImageInfoO, serviceFileIdO, disableRedirectsO);

            File2 downloaded = saveContentToLocalFile(id, content,
                    disableRetriesO.orElse(false), disableRedirectsO.orElse(false));

            Option<ServiceImageInfo> finalServiceImageInfo = serviceImageInfoO;
            if (sourceService == ChemodanService.AVIARY) {
                addExifToAviaryImage(id, downloaded,
                        aviaryOriginalMulcaIdO.getOrThrow("No original image mulca id for aviary"),
                        excludeOrientationO.orElse(false));
                // Upyachka: should be None to enable exif extraction in extractExif stage (CHEMODAN-21670)
                finalServiceImageInfo = Option.empty();
            }
            if (sourceService.isSocialNetwork()) {
                boolean isVideo = SocialDataFormatUtils.checkDataFormat(downloaded);
                if (!isVideo) {
                    // we don't want to fetch EXIF multiple times
                    finalServiceImageInfo = addCreationDateAndGeoTags(downloaded, serviceImageInfoO);
                }
            }

            stopwatch.stop();

            IncomingFile inFile = getIncomingFile(downloaded, content, maxFileSizeO, stopwatch.duration());
            return new ServiceIncomingImage(inFile,
                    serviceFileIdO,
                    sourceService,
                    finalServiceImageInfo);
        };
    }

    public Function0<ServiceIncomingFile> downloadFileFromServiceF(
            final UploadRequestId id,
            final ChemodanService sourceService,
            final Option<ServiceFileInfo> serviceFileInfoO,
            final Option<ServiceFileId> serviceFileIdO)
    {
        return () -> {
            Stopwatch stopwatch = Stopwatch.createAndStart();

            Content content = getContent(sourceService, serviceFileInfoO, serviceFileIdO, Option.empty());
            File2 downloaded = saveContentToLocalFile(id, content, false, false);

            stopwatch.stop();

            IncomingFile inFile = getIncomingFile(downloaded, content, Option.empty(), stopwatch.duration());
            return new ServiceIncomingFile(inFile,
                    serviceFileIdO,
                    sourceService,
                    serviceFileInfoO);
        };
    }

    public Function0<Boolean> patchInstaller(byte[] installer, InstallerModifierParams params) {
        return () -> {
            new InstallerModifier().patchInstaller(installer, params);
            return true;
        };
    }

    public Function0<OauthCodeResult> getAuthCodeForInstaller(String clientIp, String cookie, String clientHost) {
        return () -> new OauthCodeResult(
                oauthClient.getAuthorizationCode(
                        clientIp, cookie, OauthClient.CodeStrength.LONG, clientHost,
                        Option.of(oauthCodeTtl), Option.empty())
            );
    }

    private File2 saveContentToLocalFile(UploadRequestId id, Content content,
            boolean disableRetries, boolean disableRedirects)
    {
        File2 downloadTo = localFileManager.allocateFile(id, "download");
        return downloadToLocalFileF(content.getInputStreamSource(), downloadTo, disableRetries, disableRedirects).apply();
    }

    private IncomingFile getIncomingFile(File2 downloaded, Content content, Option<DataSize> maxFileSizeO,
            Duration duration)
    {
        DataSize downloadedSize = DataSize.fromBytes(downloaded.length());
        validateFileSize(downloadedSize);
        validateFreeDiskSpace(downloadedSize, maxFileSizeO);

        return new IncomingFile(content.getContentType(),
                downloaded.lengthO().map(DataSize::fromBytes),
                downloaded,
                Cf.list(new UploadedPartInfo(downloadedSize, duration)));
    }

    private Content getContent(ChemodanService sourceService,
            Option<? extends ServiceFileInfo> serviceFileInfo, Option<ServiceFileId> serviceFileIdO,
            Option<Boolean> disableRedirectsO)
    {
        if (sourceService.isExternal()) {
            URI photoUrl = URI.create(serviceFileInfo.get().serviceFileUrl);
            return externalResourceClient.getContent(sourceService, photoUrl, disableRedirectsO);
        } else {
            return serviceApis.download(serviceFileIdO.get(), sourceService);
        }
    }

    /**
     * Add creation date and geo-tags to image file and return their values.
     */
    private Option<ServiceImageInfo> addCreationDateAndGeoTags(File2 downloaded, Option<ServiceImageInfo> serviceImageInfoO) {
        ServiceImageInfo info = serviceImageInfoO.get();
        ExifInfo exif = ExifTool.INSTANCE.getExif(downloaded);
        if (exif.getCreationDate().isPresent() && exif.getGeoCoords().isPresent()) {
            return Option.of(new ServiceImageInfo(
                    exif.getGeoCoords(),
                    info.serviceFileUrl,
                    exif.getCreationDate()));
        }
        // Prefer EXIF
        Option<Instant> creationDate = exif.getCreationDate().orElse(info.creationTime);
        Option<ExifInfo.GeoCoords> location = exif.getGeoCoords().orElse(info.location);
        Option<LocalDateTime> ldt = creationDate.map(date -> new LocalDateTime(date, DateTimeZone.UTC));

        if (ldt.isPresent() || location.isPresent()) {
            ExifTool.INSTANCE.addGeoAndCreateTimeTags(downloaded, location, ldt);
        }

       return Option.of(new ServiceImageInfo(
                location,
                info.serviceFileUrl,
                creationDate));
    }

    private void addExifToAviaryImage(UploadRequestId request, File2 downloaded, MulcaId originalMulcaId,
            boolean excludeOrientation)
    {
        File2 originalFile = downloadFromMulcaF(request, originalMulcaId).apply().getLocalFile();
        ExifTool.INSTANCE.copyExif(originalFile, downloaded, excludeOrientation);
    }

    void setExtractExifEnabled() {
        this.extractExifEnabled = true;
    }

    void setLocalFileManager(LocalFileManager localFileManager) {
        this.localFileManager = localFileManager;
    }

}
