package ru.yandex.chemodan.uploader.registry;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.http.client.HttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.mpfs.MpfsCallbackResponse;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsHid;
import ru.yandex.chemodan.mpfs.MpfsManager;
import ru.yandex.chemodan.mpfs.MpfsResourceInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.uploader.log.ZipStreamStatLogger;
import ru.yandex.chemodan.uploader.mpfs.MpfsUtils;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.status.FileToZipInfo;
import ru.yandex.chemodan.uploader.registry.record.status.ParsedTreeResult;
import ru.yandex.chemodan.uploader.registry.record.status.ZipResourceInfo;
import ru.yandex.chemodan.util.exception.PermanentFailureException;
import ru.yandex.chemodan.util.exception.PermanentFailureWithCodeException;
import ru.yandex.chemodan.util.http.RecoverableInputStream;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaZeroSizeStids;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class ZipFolderStages {
    private static final Logger logger = LoggerFactory.getLogger(ZipFolderStages.class);

    @Autowired
    @Qualifier("mulcaHttpClientForZipFolder")
    private HttpClient mulcaHttpClientForZipFolder;
    @Autowired
    private MulcaClient mulcaClient;
    @Autowired
    private MpfsManager mpfsManager;
    @Autowired
    private MpfsClient mpfsClient;
    @Autowired
    private MulcaZeroSizeStids mulcaZeroSizeStids;

    @Value("${zip.folder.file.stream.retry.count}")
    private int zipFolderFileStreamRetryCount;
    @Value("${zip.folder.max.size}")
    private DataSize maxZipFolderSize;

    public Function0<Boolean> streamFileFromMulcaForZipArchiveF(ZipArchiveOutputStream zipArchiveOutputStream,
            UploadRequestId requestId, String normalizedPath, Option<FileToZipInfo> info, boolean auth,
            Function<FileToZipInfo, InputStream> openStream)
    {
        return () -> {
            streamFromMulca(zipArchiveOutputStream, normalizedPath, info, openStream);
            info.ifPresent(i -> {
                String name = StringUtils.substringAfterLast(normalizedPath, "/");
                ZipStreamStatLogger.logFileToZipStreamed(requestId, new MpfsHid(i.hid), name, i.mediaType, auth, i.size);
            });
            return true;
        };
    }

    public InputStream openStream(FileToZipInfo info) {
        if (mulcaZeroSizeStids.isZeroSizeStid(info.mulcaId) || info.size.toBytes() == 0) {
            // return empty input stream directly instead of call to muclagate
            logger.debug("Empty file, create empty stream directly");
            return new ByteArrayInputStream(new byte[0]);
        } else {
            String url = mulcaClient.getDownloadUri(info.mulcaId).toString();
            InputStream stream = new RecoverableInputStream(mulcaHttpClientForZipFolder, url, zipFolderFileStreamRetryCount);
            try {
                stream.read(new byte[0]);
            } catch (Exception e) {
                logger.warn(e);
            }
            return stream;
        }
    }

    void streamFromMulca(ZipArchiveOutputStream zipArchiveOutputStream, String normalizedPath,
            Option<FileToZipInfo> info, Function<FileToZipInfo, InputStream> openStream
    ) {
        try {
            String bytesStr = info.map(i -> "(length: " + i.size + ")").getOrElse("");
            logger.debug("Now streaming: '{}' {}...", normalizedPath, bytesStr);

            zipArchiveOutputStream.putArchiveEntry(new ZipArchiveEntry(normalizedPath) {{
                info.ifPresent(i -> setSize(i.size.toBytes()));
            }});

            info.map(n -> {
                long start = System.currentTimeMillis();
                try {
                    return openStream.apply(n);
                } finally {
                    logger.debug("Waiting for connection {} ms", System.currentTimeMillis() - start);
                }
            }).ifPresent(stream -> {
                try {
                    long start = System.currentTimeMillis();
                    long bytes = IoUtils.copy(stream, zipArchiveOutputStream);
                    logger.debug("Streamed {} bytes in {} ms", bytes, System.currentTimeMillis() - start);
                } finally {
                    IoUtils.closeQuietly(stream);
                }
            });

            zipArchiveOutputStream.closeArchiveEntry();

            logger.debug("Finished streaming: '{}'", normalizedPath);
        } catch (Exception e) {
            String message = "Streaming of '" + normalizedPath + "' failed";
            logger.error(message, e);
            throw new PermanentFailureException(message, e);
        }
    }

    public Function0<ParsedTreeResult> getParsedTreeF(final String mpfsFullTree) {
        return () -> {
            final ListF<MpfsResourceInfo> mpfsParsedTree = mpfsManager.parseFullTreeOutput(mpfsFullTree);

            long totalSize = mpfsParsedTree.filterMap(MpfsResourceInfo.getSizeOF()).foldLeft(0L, Cf.Long.plusF());
            if (totalSize > maxZipFolderSize.toBytes()) {
                throw new PermanentFailureWithCodeException(
                        "Total folder size is too big: " + totalSize + " bytes",
                        HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
            }

            String rootFolderName = mpfsParsedTree.find(
                    (Function1B<MpfsResourceInfo>) MpfsResourceInfo::isRelativePathRoot).single().getName();

            // sorting by isFolder is needed for proper number of files being pre-loaded
            // (as, otherwise, folders can occupy slots in executor's queue)
            // +files go first to reduce probability of error with HTTP code 200.
            // Hopefully, all zip archivers support folders in the end of zip.
            ListF<ZipResourceInfo> tree = mpfsParsedTree
                    .sorted(
                            ((Function1B<MpfsResourceInfo>) MpfsResourceInfo::isFolder).asFunction().andThenNaturalComparator()
                                    .thenComparing(
                                            ((Function<MpfsResourceInfo, String>) MpfsResourceInfo::getNormalizedPath).andThenNaturalComparator()))
                    .map(ZipResourceInfo.consFromMpfsInfoF(rootFolderName));
            Check.notEmpty(tree);

            logger.debug("Parsed tree: " + tree);

            return new ParsedTreeResult(totalSize, tree);
        };
    }

    private MpfsCallbackResponse getMpfsFullTreeResponse(MpfsRequest.ZipFolder request) {
        if (request.hash.isPresent()) {
            return mpfsClient.getFullRelativeTreePublic(request.hash.get(), true);
        } else if (request.albumKey.isPresent()) {
            return mpfsClient.getAlbumResources(request.albumKey.get().publicKey, request.albumKey.get().uid.map(MpfsUser::of));
        } else if (request.filesKey.isPresent()) {
            return mpfsClient.getFilesResources(request.filesKey.get().mpfsOid, MpfsUser.of(request.filesKey.get().uid));
        } else {
            return mpfsClient.getFullRelativeTree(MpfsUser.of(request.chemodanFile.getPassportUid()), request.chemodanFile.getPath());
        }
    }

    public CallbackResponseOption getMpfsFullTree(MpfsRequest.ZipFolder request) {
        MpfsCallbackResponse mpfsResponse = getMpfsFullTreeResponse(request);
        logger.debug("Mpfs full tree/album response: " + mpfsResponse.getResponse());
        return CallbackResponseOption.some(MpfsUtils.convert(mpfsResponse));
    }

    // for tests only
    public void setMulcaHttpClientForZipFolder(HttpClient mulcaHttpClientForZipFolder) {
        this.mulcaHttpClientForZipFolder = mulcaHttpClientForZipFolder;
    }

}
