package ru.yandex.chemodan.uploader.registry.processors;

import java.util.concurrent.atomic.AtomicReference;

import lombok.val;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.mulca.MulcaUploadManager;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.exif.ExifTool;
import ru.yandex.chemodan.uploader.log.UploaderExifLogger;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.preview.PreviewImageManager;
import ru.yandex.chemodan.uploader.preview.PreviewImageSize;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.AvatarsMeta;
import ru.yandex.chemodan.uploader.registry.RequestStatesHandler;
import ru.yandex.chemodan.uploader.registry.Stages;
import ru.yandex.chemodan.uploader.registry.record.Digests;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.DigestCalculationStatus;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.registry.record.status.MediaInfo;
import ru.yandex.chemodan.uploader.registry.record.status.PostProcessStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewDocumentStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewImageStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewVideoStatus;
import ru.yandex.chemodan.uploader.services.ServiceImageInfo;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.chemodan.util.http.ContentTypeUtils;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestDirectorUtils;
import ru.yandex.commune.uploader.registry.StageResult;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.StopDueToPrematureSuccess;
import ru.yandex.commune.uploader.util.http.Content;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public abstract class BaseUploadProcessor<T extends MpfsRequestRecord> extends RequestProcessor<T> {

    private final Logger log = LoggerFactory.getLogger(BaseUploadProcessor.class);

    protected final DynamicProperty<Long> maxSizeForImagePreview = new DynamicProperty<>("max-file-size-for-image-preview", 200L << 20); // 200mb
    protected final DocviewerClient docviewerClient;
    protected final MulcaUploadManager mulcaUploadManager;
    protected final LocalFileManager localFileManager;

    private final DynamicProperty<Boolean> sendColorData = new DynamicProperty<>("avatars-send-color-data", false);
    private final DynamicProperty<Boolean> exifLogEnabled;
    private final BleedingEdge experimentalBleedingEdge;

    BaseUploadProcessor(
            Class<T> recordClass, RequestStatesHandler listener,
            Stages stages, DocviewerClient docviewerClient,
            MulcaUploadManager mulcaUploadManager, LocalFileManager localFileManager,
            BleedingEdge experimentalBleedingEdge)
    {
        super(recordClass, listener, stages);
        this.docviewerClient = docviewerClient;
        this.mulcaUploadManager = mulcaUploadManager;
        this.localFileManager = localFileManager;
        this.experimentalBleedingEdge = experimentalBleedingEdge;
        this.exifLogEnabled = DynamicProperty.cons("uploader-exif-log-enabled", false);
    }

    void postProcessFile(Record<T> record, Content file, UidOrSpecial uidOrSpecial, PostProcessStatus pp) {
        postProcessFile(record, file, uidOrSpecial, pp, Option.empty());
    }

    void postProcessFile(Record<T> record, Content file, UidOrSpecial uidOrSpecial, PostProcessStatus pp,
            Option<ServiceImageInfo> serviceImageInfo) {
        postProcessFile(record, file, uidOrSpecial, pp, serviceImageInfo, Option.empty());
    }

    void postProcessFile(Record<T> record, Content file, UidOrSpecial uidOrSpecial, PostProcessStatus pp,
            Option<ServiceImageInfo> serviceImageInfo, Option<Digests> onTheFlyDigests)
    {
        cleanTemporaryFailureStatusIfExists(pp);
        processCallbackWithPossiblePrematureSuccess(record, pp, pp.commitFileInfo, "commitFileInfo");

        if (pp.commitFileInfo.get().isSuccess()) {
            Option<Duration> timeoutForMulcaUpload = getTimeoutForMulcaUpload(file);
            processFileMulcaUploadInfo(record, pp.fileMulcaUploadInfo,
                    stages.uploadFileToMulcaF(file.getInputStreamSource(), uidOrSpecial), timeoutForMulcaUpload);

            if (pp.fileMulcaUploadInfo.get().isSuccess()) {
                Function0<DigestCalculationStatus> calculateDigestF = onTheFlyDigests.isPresent()
                         ? () -> new DigestCalculationStatus(onTheFlyDigests.get().webDavDigestFile)
                         : stages.calculateDigestF(record.get().meta.id, file.getInputStreamSource());
                statesHandler.processSuccess(record, pp.digestCalculationStatus, calculateDigestF);

                val state = pp.digestCalculationStatus.get();
                if (state.isSuccess()) {
                    state.getResultO().flatMapO(DigestCalculationStatus::getDigestFile).ifPresent(digest ->
                            statesHandler.processSuccess(
                                    record,
                                    pp.digestMulcaUploadInfo,
                                    stages.uploadDigestToMulcaF(digest, uidOrSpecial),
                                    timeoutForMulcaUpload));

                    if (pp.digestMulcaUploadInfo.get().isSuccess()) {
                        processCallbackWithPossiblePrematureSuccess(record, pp, pp.commitFileUpload, "commitFileUpload");
                    }
                }
            }
        }

        if (pp.commitFileUpload.get().isSuccess()) {
            MulcaId fileMulcaId = pp.fileMulcaUploadInfo.get().getResultO().get().getMulcaId();
            UidOrSpecial passportUid = record.get().getRequest().chemodanFile.getUidOrSpecial();
            Option<Long> length = file.getInputStreamSource().lengthO();
            Option<String> contentType = file.getContentType();
            boolean avatarsEnabledForExperimental = passportUid
                    .uidO()
                    .map(UidOrSpecial.Uid::getPassportUid)
                    .isMatch(experimentalBleedingEdge::isOnBleedingEdge);

            extractVideoInfoIfVideo(record, file, pp.videoInfo);
            if (pp.videoInfo.get().isFinished()) {
                extractMediaInfoIfVideo(record, file, pp.mediaInfo, pp.videoInfo.get().getResultO());
            }
            generatePreviewIfDocument(record, contentType, length, fileMulcaId, passportUid, pp.previewDocumentStatus, avatarsEnabledForExperimental);

            // Extract exif info after preview generation, because hack for heif images
            if (length.isMatch(s -> s > maxSizeForImagePreview.get())) {
                RequestDirectorUtils.skip(record, pp.previewImageStatus.generateOnePreview);
                RequestDirectorUtils.skip(record, pp.previewImageStatus.previewMulcaUploadInfo);
                RequestDirectorUtils.skip(record, pp.previewImageStatus.colorData);
            } else {
                generatePreviewIfImage(record, file, fileMulcaId, passportUid, pp.previewImageStatus);
            }

            Function1V<File2> logExifF = exifLogEnabled.get()
                    ? (src) -> UploaderExifLogger.log(ExifTool.INSTANCE.getFullExifInJson(src), fileMulcaId,
                    passportUid, record.get().getRequest().chemodanFile.getPath(), length)
                    : Function1V.nop();
            extractExifInfoIfImage(record, file, serviceImageInfo, pp.exifInfo, logExifF);

            if (pp.videoInfo.get().isFinished()) {
                generatePreviewIfVideo(record, file.getContentType(),
                        Either.left(file.getInputStreamSource()),
                        passportUid, pp.previewVideoStatus, pp.videoInfo.get().getResultO(), avatarsEnabledForExperimental);
            }
            process(record, pp.antivirusResult2,
                    stages.checkWithAntivirusF(Option.of(record.get().meta.id), file.getInputStreamSource()));
        }
    }

    protected void processFileMulcaUploadInfo(Record<T> record, MutableState<MulcaUploadInfo> fileMulcaUploadInfoState,
            Function0<MulcaUploadInfo> uploadFileToMulcaF, Option<Duration> timeoutForMulcaUpload)
    {
        statesHandler.processSuccess(record, fileMulcaUploadInfoState, uploadFileToMulcaF, timeoutForMulcaUpload);
    }

    private void cleanTemporaryFailureStatusIfExists(PostProcessStatus pp) {
        if (State.StateType.TEMPORARY_FAILURE == pp.commitFileInfo.get().getType()) {
            pp.commitFileInfo.restartIfTempFailed();
        }
    }

    private void processCallbackWithPossiblePrematureSuccess(
            Record<T> record, PostProcessStatus pp,
            MutableState<CallbackResponseOption> state, String stageName)
    {
        try {
            statesHandler.processCallback(record, state, stageName);
        } catch (StopDueToPrematureSuccess ex) {
            processPrematureSuccess(record, pp);
            throw ex;
        }
    }

    protected void processPrematureSuccess(Record<T> record, PostProcessStatus pp) {
        pp.skipAllIfPrematureSuccess(record);
    }

    void generatePreviewIfDocument(
            Record<?> record, Option<String> contentType, Option<Long> sourceLength,
            MulcaId fileMulcaId, UidOrSpecial passportUid, PreviewDocumentStatus pds,
            boolean avatarsEnabled)
    {
        if (docviewerClient.isDocumentSupportedByDocviewer(contentType, sourceLength)) {
            process(record, pds.generatePreview, stages.generatePreviewForDocumentF(
                    record.get().meta.id, fileMulcaId, passportUid, contentType.get()));

            if (pds.generatePreview.get().isFinished()) {
                if (pds.generatePreview.get().getResultO().isPresent()) {
                    Function2<InputStreamSource, UidOrSpecial, Function0<MulcaUploadInfo>> upload =
                            avatarsEnabled ? (localFile, uid) -> stages.uploadFileToAvatarsF(localFile, new AtomicReference<>()) : stages::uploadFileToMulcaF;

                    statesHandler.processSuccess(record, pds.previewMulcaUploadInfo,
                            upload.apply(pds.generatePreview.get().getResultO().get(), passportUid));
                } else {
                    RequestDirectorUtils.skip(record, pds.previewMulcaUploadInfo);
                }
            }
        } else {
            RequestDirectorUtils.skip(record, pds.generatePreview);
            RequestDirectorUtils.skip(record, pds.previewMulcaUploadInfo);
        }
    }

    private void extractMediaInfoIfVideo(
            Record<?> record, Content file, MutableState<MediaInfo> mediaInfo, Option<FileInformation> videoInfo)
    {
        if (ContentTypeUtils.isVideoContentType(file.getContentType())) {
            Option<Instant> creationTimestamp = videoInfo.map(FileInformation::getCreationTime).getOrElse(Option.empty());
            process(record, mediaInfo, stages.extractMediaInfoF(creationTimestamp));
        } else {
            RequestDirectorUtils.skip(record, mediaInfo);
        }
    }

    private void extractVideoInfoIfVideo(
            Record<?> record, Content file, MutableState<FileInformation> videoInfo)
    {
        if (ContentTypeUtils.isVideoContentType(file.getContentType())) {
            process(record, videoInfo, stages.extractVideoInfoF(record.get().meta.id, file.getInputStreamSource()));
        } else {
            RequestDirectorUtils.skip(record, videoInfo);
        }
    }

    void extractVideoInfoIfVideo(
            Record<?> record, String contentType, MulcaId mulcaId, MutableState<FileInformation> videoInfo)
    {
        if (ContentTypeUtils.isVideoContentType(contentType)) {
            process(record, videoInfo, stages.extractVideoInfoF(record.get().meta.id, mulcaId));
        } else {
            RequestDirectorUtils.skip(record, videoInfo);
        }
    }

    private <T> void setResult(Record<?> record, MutableState<T> stage, T data) {
        stage.set(State.initial());
        statesHandler.processSuccess(record, stage, () -> data);
    }

    private void sendColorData(Record<?> record, MutableState<AvatarsMeta.Colors> colorData, AvatarsMeta.Colors colors) {
        if (sendColorData.get()) {
            setResult(record, colorData, colors);
        } else {
            log.info("NOT sending color data : " + colors);
        }
    }

    void generatePreviewIfVideo(Record<?> record, Option<String> contentType,
            Either<InputStreamSource, MulcaId> file, UidOrSpecial uid, PreviewVideoStatus pvs,
            Option<FileInformation> fileInfo, boolean avatarsEnabled)
    {
        RequestDirectorUtils.skip(record, pvs.colorData);
        if (ContentTypeUtils.isVideoContentType(contentType)
                && fileInfo.isMatch(i -> i.getDuration().isPresent() && i.getVideoStreamO().isPresent()))
        {
            Duration videoDuration = fileInfo.get().getDuration().get();
            Option<Dimension> dimension = fileInfo.get().getVideoStream().getDimension();

            statesHandler.process(record, pvs.generatePreview, stages
                    .generatePreviewForVideoF(record.get().meta.id, file, videoDuration, dimension));

            if (pvs.generatePreview.get().isFinished()) {
                if (pvs.generatePreview.get().getResultO().isPresent()) {
                    File2 previewDir = pvs.generatePreview.get().getResultO().get();
                    File2 previewFile = previewDir.child(PreviewVideoStatus.getPreviewFileName());
                    AtomicReference<AvatarsMeta> meta = new AtomicReference<>();
                    Function2<InputStreamSource, UidOrSpecial, Function0<MulcaUploadInfo>> upload =
                            avatarsEnabled ? (localFile, uid1) -> stages.uploadFileToAvatarsF(localFile, meta) : stages::uploadFileToMulcaF;
                    statesHandler.processSuccess(record, pvs.previewMulcaUploadInfo, upload.apply(previewFile, uid));
                    if (meta.get() != null) {
                        sendColorData(record, pvs.colorData, meta.get().getColors());
                    }

                    File2 multiplePreviewFile = previewDir.child(PreviewVideoStatus.getMultipleMontagedPreviewFileName());
                    if (multiplePreviewFile.exists()) {
                        statesHandler.processSuccess(record, pvs.multiplePreviewMulcaUploadInfo,
                                upload.apply(multiplePreviewFile, uid));
                    } else {
                        RequestDirectorUtils.skip(record, pvs.multiplePreviewMulcaUploadInfo);
                    }
                } else {
                    RequestDirectorUtils.skip(record, pvs.previewMulcaUploadInfo);
                    RequestDirectorUtils.skip(record, pvs.multiplePreviewMulcaUploadInfo);
                }
            }
        } else {
            RequestDirectorUtils.skip(record, pvs.generatePreview);
            RequestDirectorUtils.skip(record, pvs.previewMulcaUploadInfo);
            RequestDirectorUtils.skip(record, pvs.multiplePreviewMulcaUploadInfo);
        }
    }

    private void uploadImageToAvatars(Record<?> record, PreviewImageStatus pis, MulcaId stid) {
        AtomicReference<AvatarsMeta> meta = new AtomicReference<>();
        RequestDirectorUtils.skip(record, pis.generateOnePreview);
        statesHandler.processSuccess(record, pis.previewMulcaUploadInfo, stages.uploadFileToAvatarsF(stid, meta));
        if (meta.get() != null) {
            Dimension dimension = meta.get().getOrigSize().toDimension();
            sendColorData(record, pis.colorData, meta.get().getColors());
            setResult(record, pis.generateOnePreview, new GenerateImageOnePreviewResult(
                    new File2("/dev/null"),
                    ImageFormat.UNKNOWN,
                    Option.of(dimension),
                    dimension,
                    Option.empty()
            ));
        }
    }

    void generatePreviewIfImage(
            Record<?> record, Content file, MulcaId stid,
            UidOrSpecial passportUid, PreviewImageStatus pis)
    {
        RequestDirectorUtils.skip(record, pis.colorData);
        if (file.getContentType().isMatch(PreviewImageManager::isSvg)) {
            uploadImageToAvatars(record, pis, stid);
        } else if (file.getContentType().isMatch(PreviewImageManager::isSupportedMimeType)) {
            if (!file.getContentType().isMatch(PreviewImageManager::isPsd)) {
                uploadImageToAvatars(record, pis, stid);
            } else {
                uploadImageToMulca(record, pis, file, passportUid);
            }
        } else {
            RequestDirectorUtils.skip(record, pis.generateOnePreview);
            RequestDirectorUtils.skip(record, pis.previewMulcaUploadInfo);
        }
    }

    private void uploadImageToMulca(Record<?> record, PreviewImageStatus pis, Content file, UidOrSpecial passportUid) {
        process(record, pis.generateOnePreview,
                stages.generateOnePreviewForImageF(
                        record.get().meta.id,
                        file.getInputStreamSource(),
                        file.getContentType(),
                        new PreviewSizeParameter(PreviewImageSize.XXXL),
                        true, false, Option.empty()));

        if (pis.generateOnePreview.get().isFinished()) {
            if (pis.generateOnePreview.get().isSuccess()) {
                pis.generateOnePreview.get().getResultO().ifPresent(result -> statesHandler.processSuccess(record,
                        pis.previewMulcaUploadInfo, stages.uploadFileToMulcaF(result.getPreviewFile(), passportUid)));
            } else {
                RequestDirectorUtils.skip(record, pis.previewMulcaUploadInfo);
            }
        }
    }

    private void extractExifInfoIfImage(
            Record<?> record, Content file, Option<ServiceImageInfo> serviceImageInfo,
            MutableState<ExifInfo> exifInfo, Function1V<File2> logExifF)
    {
        if (file.getContentType().isMatch(ExifTool::isSupportedMimeType)) {
            val op = stages.extractExifInfoF(record.get().meta.id, file.getInputStreamSource(),
                            serviceImageInfo, file.getContentType().get(), logExifF);
            process(record, exifInfo, op);
        } else {
            RequestDirectorUtils.skip(record, exifInfo);
        }
    }

    private <F> void process(Record<?> record, MutableState<F> state, Function0<StageResult<F>> op) {
        statesHandler.process(record, state, op.asFunction(), Option.empty());
    }

    private Option<Duration> getTimeoutForMulcaUpload(Content file) {
        return file.getInputStreamSource().lengthO().map(mulcaUploadManager::getTimeoutForUpload);
    }
}
