package ru.yandex.chemodan.uploader.registry.processors;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.mulca.MulcaUploadManager;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.docviewer.DocviewerClient;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.mulca.SimultaneousMulcaUploadException;
import ru.yandex.chemodan.uploader.mulca.SimultaneousMulcaUploadManager;
import ru.yandex.chemodan.uploader.registry.RequestStatesHandler;
import ru.yandex.chemodan.uploader.registry.Stages;
import ru.yandex.chemodan.uploader.registry.record.Digests;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.PostProcessStatus;
import ru.yandex.chemodan.util.BleedingEdge;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestDirectorUtils;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.StopDueToPrematureSuccess;
import ru.yandex.commune.uploader.util.http.Content;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.ContentUtils;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.misc.time.InstantInterval;

import static ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord.UploadToDefault;

/**
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public class UploadToDefaultProcessor extends BaseUploadProcessor<UploadToDefault> {
    private final SimultaneousMulcaUploadManager simultaneousMulcaUploadManager;

    public UploadToDefaultProcessor(RequestStatesHandler listener, Stages stages,
            DocviewerClient docviewerClient, MulcaUploadManager mulcaUploadManager, LocalFileManager localFileManager,
            BleedingEdge experimentalBleedingEdge,
            SimultaneousMulcaUploadManager simultaneousMulcaUploadManager)
    {
        super(UploadToDefault.class, listener, stages, docviewerClient, mulcaUploadManager, localFileManager, experimentalBleedingEdge);
        this.simultaneousMulcaUploadManager = simultaneousMulcaUploadManager;
    }

    @Override
    protected void processTs(Record<UploadToDefault> record) {
        record.get().getStatus().userFile.get().getResultO().forEach((incomingFile) -> {
            // XXX should we use filename from request or from multipart?
            processUpload(record, incomingFile);
        });
    }

    private void processUpload(Record<UploadToDefault> record, IncomingFile incomingFile) {
        MutableState<ContentInfo> payloadInfoStatus = record.get().getStatus().payloadInfo;
        PostProcessStatus postProcess = record.get().getStatus().postProcess;
        ChemodanFile chemodanFile = record.get().getRequest().chemodanFile;

        try {
            Option<Digests> digestsO = record.get().getStatus().userFileOnTheFlyDigests.get().getResultO();

            Function0<ContentInfo> contentInfoF = digestsO.isPresent()
                    ? contentInfoWithDigestsOnTheFlyF(incomingFile, chemodanFile, digestsO.get())
                    : stages.contentInfoF(incomingFile, Option.of(chemodanFile.getPath()));

            statesHandler.processSuccess(record, payloadInfoStatus, contentInfoF);

            for (ContentInfo payloadInfo : payloadInfoStatus.get().getResultO()) {
                Content content = ContentUtils.multipartOrRaw(incomingFile);
                postProcessFile(record, ContentUtils.withType(payloadInfo.getContentType(), content),
                        chemodanFile.getUidOrSpecial(), postProcess, Option.empty(), digestsO);
            }
        } catch (StopDueToPrematureSuccess ex) {
            // premature success case handled by finalizeUpload
        } finally {
            processPrematureSuccess(record, postProcess);
        }
    }

    private Function0<ContentInfo> contentInfoWithDigestsOnTheFlyF(IncomingFile incomingFile, ChemodanFile chemodanFile,
            Digests calculatedDigests)
    {
        return () -> {
            Content content = ContentUtils.multipartOrRaw(incomingFile);
            Option<String> contentType = stages.detectMimeType(content, incomingFile.getRawFile(),
                    Option.of(chemodanFile.getPath()));

            return new ContentInfo(contentType, content.getInputStreamSource().lengthO().get(),
                    calculatedDigests.md5, calculatedDigests.sha256);
        };
    }

    @Override
    protected void processFileMulcaUploadInfo(Record<UploadToDefault> record,
            MutableState<MulcaUploadInfo> fileMulcaUploadInfoState, Function0<MulcaUploadInfo> uploadFileToMulcaF,
            Option<Duration> timeoutForMulcaUpload)
    {
        // check if the simultaneous upload started, try to wait for it
        MutableState<MulcaUploadInfo> simultaneousMulcaUploadInfo = record.get().getStatus().simultaneousMulcaUploadInfo;

        if (simultaneousMulcaUploadManager.isUploadInProgress(record.get().meta.id)
                && !simultaneousMulcaUploadInfo.get().isSkipped())
        {
            Function0<MulcaUploadInfo> waitForSimultaneousMulcaUploadInfoF =
                    () -> simultaneousMulcaUploadManager.awaitAndGetResult(record, timeoutForMulcaUpload);

            statesHandler.processSuccess(record, simultaneousMulcaUploadInfo, waitForSimultaneousMulcaUploadInfoF,
                    timeoutForMulcaUpload);

            if (simultaneousMulcaUploadInfo.get().isSuccess()) {
                statesHandler.processSuccess(record, fileMulcaUploadInfoState,
                        () -> simultaneousMulcaUploadInfo.get().getResultO().get(), timeoutForMulcaUpload);
                // simultaneous upload successful, nothing to do here anymore
                return;
            }
        }

        // either simultaneous upload didn't start or failed, fallback to normal upload
        // mark simultaneous upload as canceled first in case it will be finished later
        simultaneousMulcaUploadManager.cancelUpload(record.get().meta.id);

        if (simultaneousMulcaUploadInfo.get().getType() == State.StateType.TEMPORARY_FAILURE) {
            Instant now = new Instant();
            record.update(simultaneousMulcaUploadInfo, State.<MulcaUploadInfo>initial()
                    .failPermanently(new InstantInterval(now, now), new SimultaneousMulcaUploadException(
                            "Simultaneous upload failed, fallback to regular upload")).markSkipped());
        }

        super.processFileMulcaUploadInfo(record, fileMulcaUploadInfoState, uploadFileToMulcaF, timeoutForMulcaUpload);
    }

    @Override
    protected void processPrematureSuccess(Record<UploadToDefault> record, PostProcessStatus pp) {
        RequestDirectorUtils.skip(record, record.get().getStatus().simultaneousMulcaUploadInfo);
        super.processPrematureSuccess(record, pp);
    }
}
