package ru.yandex.chemodan.uploader.registry.processors;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.io.IOUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.http.YandexCloudRequestIdHolder;
import ru.yandex.chemodan.uploader.registry.RequestStatesHandler;
import ru.yandex.chemodan.uploader.registry.Stages;
import ru.yandex.chemodan.uploader.registry.ZipFolderStages;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord.ZipFolder;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.FileToZipInfo;
import ru.yandex.chemodan.uploader.registry.record.status.MpfsRequestStatus;
import ru.yandex.chemodan.uploader.registry.record.status.ParsedTreeResult;
import ru.yandex.chemodan.uploader.registry.record.status.StreamStatus;
import ru.yandex.chemodan.util.exception.PermanentFailureException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.uploader.registry.CachedFieldsInfo;
import ru.yandex.commune.uploader.registry.CallbackResponse;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Vsevolod Tolstopyatov (qwwdfsad)
 */
public class ZipFolderProcessor extends RequestProcessor<ZipFolder> {
    private static final Logger logger = LoggerFactory.getLogger(ZipFolderProcessor.class);
    private final ZipFolderStages zipFolderStages;
    private final DynamicProperty<Long> preloadBytes = new DynamicProperty<>("zipper-prepare-mulca-connections-for-size", 0L);
    private final DynamicProperty<Long> maxPreparedConnections = new DynamicProperty<>("zipper-max-prepared-mulca-connections", 0L);

    ZipFolderProcessor(RequestStatesHandler statesListener, ZipFolderStages zipFolderStages, Stages stages) {
        super(ZipFolder.class, statesListener, stages);
        this.zipFolderStages = zipFolderStages;
    }

    @Override
    protected void processTs(Record<ZipFolder> record) {
        final long startRequest = System.currentTimeMillis();
        MpfsRequestStatus.ZipFolder status = record.get().getStatus();
        MpfsRequest.ZipFolder request = record.get().getRequest();

        boolean haveToRetryStreaming = false;
        try {
            statesHandler.processSuccess(record, status.mpfsFullTree, () -> zipFolderStages.getMpfsFullTree(request));

            status.mpfsFullTree.get().getResultO()
                    .flatMapO(CallbackResponseOption::getCallbackResponse)
                    .map(CallbackResponse::getResponse)
                    .ifPresent(r -> statesHandler.processSuccess(record, status.parsedFullTree, zipFolderStages.getParsedTreeF(r)));

            Option<ParsedTreeResult> zipParsedTreeO = status.parsedFullTree.get().getResultO();
            ListF<StreamStatus> queue = status.multipleStream.queue;
            if (zipParsedTreeO.isPresent() && queue.isEmpty()) {
                status.multipleStream.fill(zipParsedTreeO.get().getTree());
                status.addDynamicFields(new CachedFieldsInfo(status.multipleStream));
            }

            final long startStreaming = System.currentTimeMillis();

            boolean auth = isAuthorizedRequest(request);
            ZipArchiveOutputStream zipArchiveOutputStream = status.zipOutputStream.getOrThrow(() ->
                    new PermanentFailureException("Response output stream is empty, streaming files for zip archive stopped."));
            MapF<FileToZipInfo, Function0<InputStream>> preloadedStreams = Cf.hashMap();
            Function0<Long> preloadedSize =
                    () -> preloadedStreams.keys().map(n -> n.size).reduceLeftO(DataSize::plus).getOrElse(DataSize.ZERO).toBytes();
            try {
                for (int i = 0; i < queue.size(); i++) {
                    for (int j = i; j < queue.size() && preloadedStreams.size() < maxPreparedConnections.get()
                            && preloadedSize.apply() < preloadBytes.get(); j++) {
                        queue.get(j).fileInfo.filterNot(preloadedStreams::containsKeyTs).ifPresent(n -> preloadedStreams.put(n,
                                CompletableFuture.supplyAsync(YandexCloudRequestIdHolder.supplyWithYcrid(
                                        () -> zipFolderStages.openStream(n)))::join));
                    }
                    StreamStatus ss = queue.get(i);
                    statesHandler.processSuccess(
                            record, ss.streamed,
                            zipFolderStages.streamFileFromMulcaForZipArchiveF(zipArchiveOutputStream,
                                    record.get().meta.id, ss.normalizedPath, ss.fileInfo, auth,
                                    n -> preloadedStreams.removeO(n).getOrElse(() -> zipFolderStages.openStream(n)).apply()));
                    if (!ss.streamed.get().isSuccess()) {
                        haveToRetryStreaming = State.StateType.TEMPORARY_FAILURE == ss.streamed.get().getType();
                        break;
                    }
                }
            } finally {
                preloadedStreams.values().forEach(stream -> {
                    try {
                        IOUtils.closeQuietly(stream.apply());
                    } catch (Exception e) {
                        logger.warn(e);
                    }
                });
            }
            final long now = System.currentTimeMillis();
            logger.info(
                    "Streamed {} files {} bytes in {} + {} ms",
                    queue.size(),
                    queue.flatMap(s -> s.fileInfo).map(i -> i.size).reduceLeftO(DataSize::plus).getOrElse(DataSize.ZERO).toBytes(),
                    startStreaming - startRequest,
                    now - startStreaming
            );
        } finally {
            if (!haveToRetryStreaming && status.isPreparationFinished()) {
                if (!status.isPreparationFailed()) {
                    IoUtils.closeQuietly(status.zipOutputStream.getOrNull());
                }

                logger.debug("Release lock for record: " + record.get().meta.id);
                status.latch.countDown();
            }
        }
    }

    private boolean isAuthorizedRequest(MpfsRequest.ZipFolder request) {
        return request.isForPrivateFolder() || request.isForFiles()
                || (request.isForAlbum() && request.albumKey.get().uid.isPresent());
    }
}
