package ru.yandex.chemodan.uploader.registry.record;

import java.net.URI;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.dom4j.Element;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.mpfs.MpfsAlbumKey;
import ru.yandex.chemodan.mpfs.MpfsFilesKey;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.services.ServiceFileId;
import ru.yandex.chemodan.uploader.services.ServiceFileInfo;
import ru.yandex.chemodan.uploader.services.ServiceImageInfo;
import ru.yandex.commune.uploader.registry.UploadRequest;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * Simple structures reflecting all possible types of MPFS requests with their arguments.
 *
 * These classes know nothing about how the requests will be fulfilled within the uploader.
 *
 * @author vavinov
 */
public abstract class MpfsRequest extends UploadRequest {
    @JsonProperty
    public final ApiVersion apiVersion;          // better typing
    @JsonProperty
    public final ChemodanFile chemodanFile;    // XXX should not be present in UploadToService and ListArchive

    private MpfsRequest(ApiVersion apiVersion,
            Option<URI> callbackUri,
            ChemodanFile chemodanFile,
            Option<DataSize> maxFileSize,
            Option<String> yandexCloudRequestId)
    {
        super(callbackUri, maxFileSize, yandexCloudRequestId);
        this.apiVersion = apiVersion;
        this.chemodanFile = chemodanFile;
    }

    public static class UploadToDefault extends MpfsRequest {
        @JsonProperty
        public final Option<DataSize> uploadMaxSpeedBps;

        @JsonCreator
        public UploadToDefault(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId,
                @JsonProperty("uploadMaxSpeedBps") Option<DataSize> uploadMaxSpeedBps)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.uploadMaxSpeedBps = uploadMaxSpeedBps;
        }

        public UploadToDefault(
                ApiVersion apiVersion,
                ChemodanFile chemodanFile,
                Option<URI> callbackUri,
                Option<DataSize> maxFileSize,
                Option<String> yandexCloudRequestId)
        {
            this(apiVersion, chemodanFile, callbackUri, maxFileSize, yandexCloudRequestId, Option.empty());
        }

        @Override
        public void xmlize(XmlWriter xw) {
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            if (uploadMaxSpeedBps.isPresent()) {
                xw.addAttribute("upload-max-speed-bps", uploadMaxSpeedBps.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class UploadToService extends MpfsRequest {
        @JsonProperty
        public final ChemodanService targetService;
        @JsonProperty
        public final Element metaData;

        @JsonCreator
        public UploadToService(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("targetService") ChemodanService targetService,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("metaData") Element metaData,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.targetService = targetService;
            this.metaData = metaData;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("target-service", targetService.name().toLowerCase());
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }

        }
    }

    public static class PatchAtDefault extends MpfsRequest {
        @JsonProperty
        public final MulcaId originalFile;
        @JsonProperty
        public final String originalMd5;

        @JsonCreator
        public PatchAtDefault(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("originalFile") MulcaId originalFile,
                @JsonProperty("originalMd5") String originalMd5,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.originalFile = originalFile;
            this.originalMd5 = originalMd5;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("original-md5", originalMd5);
            // TODO expected-patched-md5
            xw.addAttribute("original-mulca-id", originalFile.getStidCheckNoPart());
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class ListArchive extends MpfsRequest {
        @JsonProperty
        public final MulcaId originalFile;

        @JsonCreator
        public ListArchive(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("originalFile") MulcaId originalFile,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.originalFile = originalFile;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("original-mulca-id", originalFile.getStidCheckNoPart());
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
        }
    }

    public static class ExtractArchive extends MpfsRequest {
        @JsonProperty
        public final Option<String> fileToExtract;
        @JsonProperty
        public final ChemodanService sourceService;
        @JsonProperty
        public final Option<ServiceFileId> serviceFileId;
        @JsonProperty
        public final Option<ServiceFileInfo> serviceFileInfo;

        @JsonCreator
        public ExtractArchive(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("sourceService") ChemodanService sourceService,
                @JsonProperty("serviceFileId") Option<ServiceFileId> serviceFileId,
                @JsonProperty("serviceFileInfo") Option<ServiceFileInfo> serviceFileInfo,
                @JsonProperty("fileToExtract") Option<String> fileToExtract,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.fileToExtract = fileToExtract;
            this.sourceService = sourceService;
            this.serviceFileId = serviceFileId;
            this.serviceFileInfo = serviceFileInfo;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("source-service", sourceService.toString());
            if (serviceFileId.isPresent()) {
                xw.addAttribute("service-file-id", serviceFileId.get().n);
            }
            if (serviceFileInfo.isPresent()) {
                xw.addAttribute("service-file-url", serviceFileInfo.get().serviceFileUrl);
            }
            if (fileToExtract.isPresent()) {
                xw.addAttribute("file-to-extract", fileToExtract.get());
            }
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class ExtractFileFromArchive extends MpfsRequest {
        @JsonProperty
        public final String fileToExtract;
        @JsonProperty
        public final ChemodanService sourceService;
        @JsonProperty
        public final Option<ServiceFileId> serviceFileId;
        @JsonProperty
        public final Option<ServiceFileInfo> serviceFileInfo;

        @JsonCreator
        public ExtractFileFromArchive(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("sourceService") ChemodanService sourceService,
                @JsonProperty("serviceFileId") Option<ServiceFileId> serviceFileId,
                @JsonProperty("serviceFileInfo") Option<ServiceFileInfo> serviceFileInfo,
                @JsonProperty("fileToExtract") String fileToExtract,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.fileToExtract = fileToExtract;
            this.sourceService = sourceService;
            this.serviceFileId = serviceFileId;
            this.serviceFileInfo = serviceFileInfo;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("source-service", sourceService.toString());
            if (serviceFileId.isPresent()) {
                xw.addAttribute("service-file-id", serviceFileId.get().n);
            }
            if (serviceFileInfo.isPresent()) {
                xw.addAttribute("service-file-url", serviceFileInfo.get().serviceFileUrl);
            }
            xw.addAttribute("file-to-extract", fileToExtract);
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class ConvertToMsOoxmlFormat extends MpfsRequest {
        @JsonProperty
        public final Option<String> contentType;
        @JsonProperty
        public final Option<String> archivePath;
        @JsonProperty
        public final ChemodanService sourceService;
        @JsonProperty
        public final Option<ServiceFileId> serviceFileId;
        @JsonProperty
        public final Option<ServiceFileInfo> serviceFileInfo;

        @JsonCreator
        public ConvertToMsOoxmlFormat(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("sourceService") ChemodanService sourceService,
                @JsonProperty("serviceFileId") Option<ServiceFileId> serviceFileId,
                @JsonProperty("serviceFileInfo") Option<ServiceFileInfo> serviceFileInfo,
                @JsonProperty("contentType") Option<String> contentType,
                @JsonProperty("archivePath") Option<String> archivePath,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.contentType = contentType;
            this.archivePath = archivePath;
            this.sourceService = sourceService;
            this.serviceFileId = serviceFileId;
            this.serviceFileInfo = serviceFileInfo;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("source-service", sourceService.toString());
            if (serviceFileId.isPresent()) {
                xw.addAttribute("service-file-id", serviceFileId.get().n);
            }
            if (serviceFileInfo.isPresent()) {
                xw.addAttribute("service-file-url", serviceFileInfo.get().serviceFileUrl);
            }
            if (contentType.isPresent()) {
                xw.addAttribute("content-type", contentType.get());
            }
            if (archivePath.isPresent()) {
                xw.addAttribute("archive-path", archivePath.get());
            }
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class GeneratePreview extends MpfsRequest {
        @JsonProperty
        public final MulcaId originalFile;
        @JsonProperty
        public final PreviewSizeParameter size;
        @JsonProperty
        public final Boolean crop;
        @JsonProperty
        public final Option<Integer> quality;

        @JsonCreator
        public GeneratePreview(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("originalFile") MulcaId originalFile,
                @JsonProperty("size") PreviewSizeParameter size,
                @JsonProperty("crop") Boolean crop,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId,
                @JsonProperty("quality") Option<Integer> quality)
        {
            super(apiVersion, Option.empty(), ChemodanFile.FAKE, Option.empty(), yandexCloudRequestId);
            this.originalFile = originalFile;
            this.size = size;
            this.crop = crop != null ? crop : Boolean.FALSE;
            this.quality = quality;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }
    }

    public static class RegeneratePreview extends MpfsRequest {
        @JsonProperty
        public final MulcaId originalFile;
        @JsonProperty
        public final String mimeType;
        @JsonProperty
        public final Option<DataSize> size;
        @JsonProperty
        public final Option<String> fileName;

        @JsonCreator
        public RegeneratePreview(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("originalFile") MulcaId originalFile,
                @JsonProperty("mimeType") String mimeType,
                @JsonProperty("size") Option<DataSize> size,
                @JsonProperty("fileName") Option<String> fileName,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, Option.empty(), ChemodanFile.FAKE, Option.empty(), yandexCloudRequestId);
            this.originalFile = originalFile;
            this.mimeType = mimeType;
            this.size = size;
            this.fileName = fileName;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }
    }

    public static class ZipFolder extends MpfsRequest {
        public final Option<String> hash;
        public final Option<MpfsAlbumKey> albumKey;
        public final Option<MpfsFilesKey> filesKey;

        public ZipFolder(ApiVersion apiVersion, ChemodanFile chemodanFile, Option<String> hash,
                Option<MpfsAlbumKey> albumKey, Option<MpfsFilesKey> filesKey, Option<String> yandexCloudRequestId)
        {
            super(apiVersion, Option.empty(), chemodanFile, Option.empty(), yandexCloudRequestId);
            Validate.isTrue(!hash.isPresent() || !albumKey.isPresent());
            this.hash = hash;
            this.albumKey = albumKey;
            this.filesKey = filesKey;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }

        public boolean isForAlbum() {
            return albumKey.isPresent();
        }

        public boolean isForPublicFolder() {
            return hash.isPresent();
        }

        public boolean isForPrivateFolder() {
            return !albumKey.isPresent() && !hash.isPresent();
        }

        public boolean isForFiles() {
            return filesKey.isPresent();
        }
    }

    public static class PublishToService extends MpfsRequest {
        @JsonProperty
        public final ChemodanService targetService;
        @JsonProperty
        public final MulcaId diskFile;
        @JsonProperty
        public final Element metaData;

        @JsonCreator
        public PublishToService(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("targetService") ChemodanService targetService,
                @JsonProperty("diskFile") MulcaId diskFile,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("metaData") Element metaData,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.targetService = targetService;
            this.diskFile = diskFile;
            this.metaData = metaData;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            xw.addAttribute("original-mulca-id", diskFile.getStidCheckNoPart());
            xw.addAttribute("target-service", targetService.name().toLowerCase());
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static class UploadFromService extends MpfsRequest {

        @JsonProperty
        public final Option<Boolean> disableRetries;
        @JsonProperty
        public final Option<Boolean> disableRedirects;
        @JsonProperty
        public final ChemodanService sourceService;
        @JsonProperty
        public final Option<ServiceFileId> serviceFileId;
        @JsonProperty
        public final Option<ServiceImageInfo> serviceImageInfo;
        @JsonProperty
        public final Option<MulcaId> aviaryOriginalMulcaId;
        @JsonProperty
        public final Option<Boolean> excludeOrientation;

        @JsonCreator
        public UploadFromService(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("sourceService") ChemodanService sourceService,
                @JsonProperty("serviceFileId") Option<ServiceFileId> serviceFileId,
                @JsonProperty("chemodanFile") ChemodanFile chemodanFile,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("disableRetries") Option<Boolean> disableRetries,
                @JsonProperty("disableRedirects") Option<Boolean> disableRedirects,
                @JsonProperty("maxFileSize") Option<DataSize> maxFileSize,
                @JsonProperty("serviceImageInfo") Option<ServiceImageInfo> serviceImageInfo,
                @JsonProperty("aviaryOriginalMulcaId") Option<MulcaId> aviaryOriginalMulcaId,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId,
                @JsonProperty("excludeOrientation") Option<Boolean> excludeOrientation)
        {
            super(apiVersion, callbackUri, chemodanFile, maxFileSize, yandexCloudRequestId);
            this.disableRetries = disableRetries;
            this.disableRedirects = disableRedirects;
            this.sourceService = sourceService;
            this.serviceFileId = serviceFileId;
            this.serviceImageInfo = serviceImageInfo;
            this.aviaryOriginalMulcaId = aviaryOriginalMulcaId;
            this.excludeOrientation = excludeOrientation;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            if (callbackUri.isPresent()) {
                xw.addAttribute("callback", callbackUri.get());
            }
            xmlizeChemodanFile(xw, chemodanFile);
        }
    }

    public static void xmlizeChemodanFile(XmlWriter xw, ChemodanFile chemodanFile) {
        xw.startElement("chemodan-file-attributes");

        UidOrSpecial uidOrSpecial = chemodanFile.getUidOrSpecial();
        if (uidOrSpecial.specialO().isPresent()) {
            xw.addAttribute("special-uid", uidOrSpecial.specialO().get().getSpecial());
        } else {
            xw.addAttribute("uid", uidOrSpecial.uidO().get().getPassportUid().getUid());
        }

        xw.addAttribute("file-id", chemodanFile.getUniqueFileId());
        xw.addAttribute("path", chemodanFile.getPath());
        xw.endElement();
    }

    public static class ExportPhotos extends MpfsRequest {
        @JsonProperty
        public final ListF<MulcaId> photoPreviewMulcaIds;
        @JsonProperty
        public final String profileId;
        @JsonProperty
        public final String albumId;
        @JsonProperty
        public final ChemodanService targetService;

        @JsonCreator
        public ExportPhotos(
                @JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("photoPreviewMulcaIds") ListF<MulcaId> photoPreviewMulcaIds,
                @JsonProperty("profileId") String profileId,
                @JsonProperty("albumKey") String albumId,
                @JsonProperty("targetService") ChemodanService targetService,
                @JsonProperty("callbackUri") Option<URI> callbackUri,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, callbackUri, ChemodanFile.FAKE, Option.empty(), yandexCloudRequestId);
            this.photoPreviewMulcaIds = photoPreviewMulcaIds;
            this.profileId = profileId;
            this.albumId = albumId;
            this.targetService = targetService;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }
    }

    public static class MarkMulcaIdsForRemove extends MpfsRequest {

        @JsonProperty
        public final ListF<MulcaId> mulcaIds;

        @JsonCreator
        public MarkMulcaIdsForRemove(@JsonProperty("apiVersion") ApiVersion apiVersion,
                @JsonProperty("mulcaIds") ListF<MulcaId> mulcaIds,
                @JsonProperty("yandexCloudRequestId") Option<String> yandexCloudRequestId)
        {
            super(apiVersion, Option.empty(), ChemodanFile.FAKE, Option.empty(), yandexCloudRequestId);
            this.mulcaIds = mulcaIds;
        }

        @Override
        public void xmlize(XmlWriter xw) {
        }
    }
}
