package ru.yandex.chemodan.uploader.registry.record;

import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.uploader.av.AntivirusResult;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.registry.AvatarsMeta;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo;
import ru.yandex.chemodan.uploader.registry.record.status.ExifInfo.GeoCoords;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.registry.record.status.MediaInfo;
import ru.yandex.chemodan.uploader.registry.record.status.PostProcessStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewDocumentStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewImageStatus;
import ru.yandex.chemodan.uploader.registry.record.status.PreviewVideoStatus;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.uploader.registry.CallbackResponse;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * http://wiki.yandex-team.ru/Pochta/chemodan/uploader/api#h108377-5
 *
 * @author vavinov
 * @since ApiVersion.V_0_2
 */
public class Xmlizer2 {
    private static final String MULCA_ID_ATTR = "mulca-id";

    private static final BenderConfiguration benderConfig = BenderConfiguration.defaultConfiguration();
    private static final BenderParserSerializer<ArchiveListing> archiveSerializer = Bender.cons(ArchiveListing.class, benderConfig);
    private static final BenderJsonSerializer<AvatarsMeta.Colors> colorDataSerializer = Bender.jsonSerializer(AvatarsMeta.Colors.class, benderConfig);

    public static void xmlizePostProcessStatus(XmlWriter xw, PostProcessStatus pp) {
        pp.commitFileInfo.xmlize(xw, "commit-file-info", callbackResponseOptionF());
        pp.fileMulcaUploadInfo.xmlize(xw, "mulca-file", mulcaUploadInfoF());
        pp.digestMulcaUploadInfo.xmlize(xw, "mulca-digest", mulcaUploadInfoF());
        pp.commitFileUpload.xmlize(xw, "commit-file-upload", callbackResponseOptionF());
        xmlizePreviewDocumentStatus(xw, pp.previewDocumentStatus);
        xmlizePreviewImageStatus(xw, pp.previewImageStatus);
        xmlizeVideo(xw, pp.videoInfo, pp.previewVideoStatus);
        pp.exifInfo.xmlize(xw, "exif-info", exifInfoF());
        pp.mediaInfo.xmlize(xw, "media-info", mediaInfoF());
        pp.antivirusResult2.xmlize(xw, "antivirus", antivirusResultF());
    }

    public static String xmlizeArchiveList(ArchiveListing listing) {
        return new String(archiveSerializer.getSerializer().serializeXml(listing));
    }

    public static void xmlizeArchiveList(ArchiveListing listing, XmlWriter xw) {
        archiveSerializer.getSerializer().serializeXml(listing, xw);
    }

    private static void xmlizePreviewDocumentStatus(XmlWriter xw, PreviewDocumentStatus pds) {
        if (pds.previewMulcaUploadInfo.get().getResultO().isPresent()) {
            pds.previewMulcaUploadInfo.xmlize(xw, "preview", mulcaUploadInfoF());
        }
    }

    private static void xmlizePreviewImageStatus(XmlWriter xw, PreviewImageStatus pis) {
        if (pis.previewMulcaUploadInfo.get().getResultO().isPresent()) {
            pis.generateOnePreview.xmlize(xw, "generate-image-one-preview", generateOneImagePreviewResultF());
            pis.previewMulcaUploadInfo.xmlize(xw, "preview-image-mulca-upload", mulcaUploadInfoF());
            if (pis.colorData.get().getResultO().isPresent()) {
                pis.colorData.xmlize(xw, "color-data", colorDataResultF());
            }
        }
    }

    private static void xmlizeVideo(XmlWriter xw, MutableState<FileInformation> videoInfo, PreviewVideoStatus pvs) {
        if (pvs.previewMulcaUploadInfo.get().getResultO().isPresent()) {
            videoInfo.xmlize(xw, "video-info", videoInfoF());
            pvs.previewMulcaUploadInfo.xmlize(xw, "preview-video-mulca-upload", mulcaUploadInfoF());
            if (pvs.multiplePreviewMulcaUploadInfo.get().getResultO().isPresent()) {
                pvs.multiplePreviewMulcaUploadInfo
                        .xmlize(xw, "multiple-preview-video-mulca-upload", mulcaUploadInfoF());
            }
            if (pvs.colorData.get().getResultO().isPresent()) {
                pvs.colorData.xmlize(xw, "color-data", colorDataResultF());
            }
        }
    }

    private static Function2V<XmlWriter, ExifInfo> exifInfoF() {
        return (xw, info) -> {
            for (Instant date : info.getCreationDate()) {
                xw.addAttribute("creation-date", ISODateTimeFormat.dateTimeNoMillis().print(date));
            }
            for (GeoCoords coords : info.getGeoCoords()) {
                xw.addAttribute("latitude", coords.getLatitude());
                xw.addAttribute("longitude", coords.getLongitude());
            }
        };
    }

    private static Function2V<XmlWriter, MediaInfo> mediaInfoF() {
        return (xw, info) -> {
            for (Instant date : info.getCreationDate()) {
                xw.addAttribute("creation-date", ISODateTimeFormat.dateTimeNoMillis().print(date));
            }
        };
    }

    private static Function2V<XmlWriter, FileInformation> videoInfoF() {
        return (xw, videoInfo) -> {
            byte[] serializedJson = FileInformation.PS.getSerializer().serializeJson(videoInfo);
            xw.addCharacters(new String(serializedJson, CharsetUtils.UTF8_CHARSET));
        };
    }

    private static Function2V<XmlWriter, MulcaUploadInfo> mulcaUploadInfoF() {
        return (xw, o) -> xw.addAttribute(MULCA_ID_ATTR, o.getMulcaId().getStidCheckNoPart());
    }

    private static Function2V<XmlWriter, CallbackResponseOption> callbackResponseOptionF() {
        return (xw, o) -> {
            for (CallbackResponse response : o.getCallbackResponse()) {
                xw.addAttribute("status-line", response.getStatusLine());
                if (response.getUri().isPresent()) {
                    xw.addAttribute("url", response.getUri().get());
                }
            }
        };
    }

    private static Function2V<XmlWriter, AvatarsMeta.Colors> colorDataResultF() {
        return (xw, cd) -> xw.addAttribute("data", colorDataSerializer.serializeJson(cd));
    }

    private static Function2V<XmlWriter, AntivirusResult> antivirusResultF() {
        return (xw, r) -> {
            xw.addAttribute("result-ext", r.xmlName());
            // XXX: legacy
            xw.addAttribute("result", r == AntivirusResult.HEALTHY || r == AntivirusResult.UNKNOWN);
        };
    }

    public static Function2V<XmlWriter, GenerateImageOnePreviewResult> generateOneImagePreviewResultF() {
        return (xw, r) -> {
            xw.addAttribute("format", r.getPreviewFormat().name().toLowerCase());
            xw.addAttribute("width", r.getPreviewSize().getWidth());
            xw.addAttribute("height", r.getPreviewSize().getHeight());
            if (r.getOrigSize().isPresent()) {
                xw.addAttribute("original-width", r.getOrigSize().get().getWidth());
                xw.addAttribute("original-height", r.getOrigSize().get().getHeight());
            }
            if (r.getRotateAngle().isPresent()) {
                xw.addAttribute("rotate-angle", r.getRotateAngle().get().getAngle());
            }
        };
    }

    public static Function2V<XmlWriter, IncomingFile> xmlizeIncomingFileF() {
        return (xw, result) -> {
            for (String contentType : result.getContentType()) {
                xw.addAttribute("content-type", contentType);
            }
            for (DataSize contentLength : result.getContentLength()) {
                xw.addAttribute("content-length", contentLength.toBytes());
            }
            xw.addAttribute("local-file", result.getRawFile());
        };
    }

    public static Function2V<XmlWriter, ContentInfo> xmlizeContentInfoF() {
        return (xw, contentInfo) -> {
            for (String contentType : contentInfo.getContentType()) {
                xw.addAttribute("content-type", contentType);
            }
            xw.addAttribute("content-length", contentInfo.getContentLength());
            xw.addAttribute("md5", contentInfo.getMd5());
            xw.addAttribute("sha256", contentInfo.getSha256());
        };
    }

    public static Function2V<XmlWriter, Boolean> exportPhotoF(final String attrName, final int index) {
        return (xw, o) -> {
            xw.addAttribute(attrName, o);
            xw.addAttribute("ind", index);
        };
    }

}
