package ru.yandex.chemodan.uploader.registry.record.status;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.joda.time.Instant;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.uploader.log.DescribableResult;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;

/**
 * @author akirakozov
 */
public class ExifInfo extends DefaultObject implements DescribableResult {
    private static final String NONE_VALUE = "none";

    @JsonProperty
    private final Option<Instant> creationDate;

    @JsonProperty
    private final Option<GeoCoords> geoCoords;

    public boolean isPresent() {
        return geoCoords.isPresent() || creationDate.isPresent();
    }

    public static class GeoCoords extends DefaultObject {
        @JsonProperty
        private final double latitude;
        @JsonProperty
        private final double longitude;

        public GeoCoords(
                @JsonProperty("latitude") double latitude,
                @JsonProperty("longitude") double longitude)
        {
            Validate.isTrue(Math.abs(latitude) <= 90.0, "latitude out of range -90..90: ", latitude);
            Validate.isTrue(Math.abs(longitude) <= 180.0, "longitude out of range -180..180: ", longitude);

            this.latitude = latitude;
            this.longitude = longitude;
        }

        public double getLatitude() {
            return latitude;
        }

        public double getLongitude() {
            return longitude;
        }
    }

    @JsonCreator
    public ExifInfo(
            @JsonProperty("creationDate") Option<Instant> creationDate,
            @JsonProperty("geoCoords") Option<GeoCoords> geoCoords)
    {
        this.creationDate = creationDate;
        this.geoCoords = geoCoords;
    }

    public Option<Instant> getCreationDate() {
        return creationDate;
    }

    public Option<GeoCoords> getGeoCoords() {
        return geoCoords;
    }

    @Override
    public MapF<String, String> describe() {
        String creationDate = this.creationDate.isPresent() ?
                ISODateTimeFormat.dateTimeNoMillis().print(this.creationDate.get()) : NONE_VALUE;
        String latitude = geoCoords.isPresent() ?
                Double.toString(geoCoords.get().getLatitude()) : NONE_VALUE;
        String longitude = geoCoords.isPresent() ?
                Double.toString(geoCoords.get().getLongitude()) : NONE_VALUE;

        return Cf.map("creation-date", creationDate)
                .plus1("latitude", latitude)
                .plus1("longitude", longitude);
    }
}
