package ru.yandex.chemodan.uploader.registry.record.status;

import java.util.concurrent.CountDownLatch;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.jetbrains.annotations.Nullable;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.uploader.ExtractedFile;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.registry.record.Digests;
import ru.yandex.chemodan.uploader.registry.record.Xmlizer2;
import ru.yandex.chemodan.uploader.services.ServiceIncomingFile;
import ru.yandex.chemodan.uploader.services.ServiceIncomingImage;
import ru.yandex.chemodan.uploader.services.ServicePublishResult;
import ru.yandex.chemodan.util.exception.PermanentFailureWithCodeException;
import ru.yandex.commune.archive.ArchiveListing;
import ru.yandex.commune.uploader.registry.CallbackResponseOption;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.RequestStatusRegistry;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRequestStatus;
import ru.yandex.commune.uploader.util.http.ContentInfo;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * Simple structures that hold status of MpfsRequest completion (named the same as
 * the corresponding MpfsRequest-s).
 *
 * Mutable because it's too much hassle writing with- methods (maybe later).
 *
 * TODO remove most @Nullable-s (they are for migration to new format only)
 *
 * @author vavinov
 * @author akirakozov
 * @author ssytnik
 */
public abstract class MpfsRequestStatus extends UploadRequestStatus {

    private MpfsRequestStatus(@Nullable State<CallbackResponseOption> commitFinal) {
        super(commitFinal);
    }

    public static void registerStatuses() {
        RequestStatusRegistry.registerStatusClass(UploadToDefault.class);
        RequestStatusRegistry.registerStatusClass(UploadToService.class);
        RequestStatusRegistry.registerStatusClass(UploadFromService.class);
        RequestStatusRegistry.registerStatusClass(PatchAtDefault.class);
        RequestStatusRegistry.registerStatusClass(PublishToService.class);
        RequestStatusRegistry.registerStatusClass(ListArchive.class);
        RequestStatusRegistry.registerStatusClass(ExtractArchive.class);
        RequestStatusRegistry.registerStatusClass(ExtractFileFromArchive.class);
        RequestStatusRegistry.registerStatusClass(GeneratePreview.class);
        RequestStatusRegistry.registerStatusClass(ZipFolder.class);
        RequestStatusRegistry.registerStatusClass(ConvertToMsOoxmlFormat.class);
    }

    public abstract boolean isUploadingToDisk();

    public boolean isNeedToRollbackChanges() {
        for (PostProcessStatus pps : getPostProcessStatusO()) {
            return pps.isNeedToRollbackChanges();
        }
        return false;
    }

    public Option<PostProcessStatus> getPostProcessStatusO() {
        return Option.empty();
    }

    @Override
    public boolean isLocallyUploadedFileFinished() {
        throw new UnsupportedOperationException();
    }

    public boolean isCommitFileUploadFinished() {
        return getPostProcessStatusO()
                .getOrThrow(() -> new UnsupportedOperationException())
                .commitFileUpload.get().isFinished();
    }

    private static Function2V<XmlWriter, ServicePublishResult> xmlizeServicePublishResult() {
        return new Function2V<XmlWriter, ServicePublishResult>() {
            public void apply(XmlWriter xww, ServicePublishResult result) {
                xww.addAttribute("service-file-id", result.serviceFileId.toSerializedString());
                xww.addAttribute("uid", result.serviceFileId.uid);
                xww.addAttribute("file-id", result.serviceFileId.n);
                xww.textElement("summary", result.serviceSpecificSummary);
            }
        };
    }

    public static class UploadToDefault extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<IncomingFile> userFile;
        @JsonProperty
        public final MutableState<ContentInfo> payloadInfo;
        @JsonProperty
        public final PostProcessStatus postProcess;
        @JsonProperty
        public final MutableState<Digests> userFileOnTheFlyDigests;
        @JsonProperty
        public final MutableState<MulcaUploadInfo> simultaneousMulcaUploadInfo;

        public UploadToDefault() {
            this(State.initial(), State.initial(), new PostProcessStatus(), State.initial(),
                    State.<Digests>initial().skipSuccess(), State.<MulcaUploadInfo>initial().skipSuccess());
        }

        @JsonCreator
        public UploadToDefault(
                @JsonProperty("userFile") State<IncomingFile> userFile,
                @JsonProperty("payloadInfo") State<ContentInfo> payloadInfo,
                @JsonProperty("postProcess") PostProcessStatus postProcess,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal,
                @Nullable @JsonProperty("userFileOnTheFlyDigests") State<Digests> userFileOnTheFlyDigests,
                @Nullable @JsonProperty("simultaneousMulcaUploadInfo") State<MulcaUploadInfo> simultaneousMulcaUploadInfo)
        {
            super(commitFinal);
            this.userFile = MutableState.cons(userFile);
            this.payloadInfo = MutableState.cons(payloadInfo);
            this.postProcess = postProcess;
            this.userFileOnTheFlyDigests = MutableState.cons(Option.ofNullable(userFileOnTheFlyDigests)
                    .getOrElse(State.<Digests>initial().skipSuccess()));
            this.simultaneousMulcaUploadInfo = MutableState.cons(Option.ofNullable(simultaneousMulcaUploadInfo)
                    .getOrElse(State.<MulcaUploadInfo>initial().skipSuccess()));
        }

        @Override
        public Option<MutableState<IncomingFile>> waitForExternalField() {
            return Option.of(userFile);
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public boolean isLocallyUploadedFileFinished() {
            return userFile.get().isFinished();
        }

        @Override
        public Option<PostProcessStatus> getPostProcessStatusO() {
            return Option.of(postProcess);
        }

        @Override
        public void xmlize(XmlWriter xw) {
            userFile.xmlize(xw, "incoming-http", Xmlizer2.xmlizeIncomingFileF());
            payloadInfo.xmlize(xw, "incoming-file", Xmlizer2.xmlizeContentInfoF());
            Xmlizer2.xmlizePostProcessStatus(xw, postProcess);
        }
    }

    public static class UploadToService extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<IncomingFile> userFile;
        @JsonProperty
        public final MutableState<ContentInfo> payloadInfo;
        @JsonProperty
        public final MutableState<ServicePublishResult> publishResult;

        public UploadToService() {
            this(State.initial(), State.initial(), State.initial(), State.initial());
        }

        @JsonCreator
        public UploadToService(
                @JsonProperty("userFile") State<IncomingFile> userFile,
                @JsonProperty("payloadInfo") State<ContentInfo> payloadInfo,
                @JsonProperty("publishResult") State<ServicePublishResult> publishResult,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.userFile = MutableState.cons(userFile);
            this.payloadInfo = MutableState.cons(payloadInfo);
            this.publishResult = MutableState.cons(publishResult);
        }

        @Override
        public Option<MutableState<IncomingFile>> waitForExternalField() {
            return Option.of(userFile);
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            userFile.xmlize(xw, "incoming-http", Xmlizer2.xmlizeIncomingFileF());

            payloadInfo.xmlize(xw, "incoming-file", new Function2V<XmlWriter, ContentInfo>() {
                public void apply(XmlWriter xw, ContentInfo contentInfo) {
                    for (String contentType : contentInfo.getContentType()) {
                        xw.addAttribute("content-type", contentType);
                    }
                    xw.addAttribute("content-length", contentInfo.getContentLength());
                    xw.addAttribute("md5", contentInfo.getMd5());
                    xw.addAttribute("sha256", contentInfo.getSha256());
                }
            });

            publishResult.xmlize(xw, "publish", xmlizeServicePublishResult());

        }
    }

    public static class PatchAtDefault extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<String> expectedPatchedMd5;
        @JsonProperty
        public final MutableState<IncomingFile> incomingPatch;
        @JsonProperty
        public final MutableState<UploadedMulcaFile> originalFile2;
        @JsonProperty
        public final MutableState<File2> patchedFile;
        @JsonProperty
        public final MutableState<ContentInfo> patchedPayloadInfo;
        @JsonProperty
        public final PostProcessStatus postProcess;

        public PatchAtDefault() {
            this(State.initial(), State.initial(), State.initial(), State.initial(), State.initial(),
                    new PostProcessStatus(), State.initial());
        }

        @JsonCreator
        public PatchAtDefault(
                @JsonProperty("expectedPatchedMd5") State<String> expectedPatchedMd5,
                @JsonProperty("incomingPatch") State<IncomingFile> incomingPatch,
                @JsonProperty("originalFile2") State<UploadedMulcaFile> originalFile2,
                @JsonProperty("patchedFile") State<File2> patchedFile,
                @JsonProperty("patchedPayloadInfo") State<ContentInfo> patchedPayloadInfo,
                @JsonProperty("postProcess") PostProcessStatus postProcess,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.expectedPatchedMd5 = MutableState.cons(expectedPatchedMd5);
            this.incomingPatch = MutableState.cons(incomingPatch);
            this.originalFile2 = MutableState.fromNullable(originalFile2);
            this.patchedFile = MutableState.cons(patchedFile);
            this.patchedPayloadInfo = MutableState.fromNullable(patchedPayloadInfo);
            this.postProcess = postProcess;
        }

        @Override
        public Option<MutableState<IncomingFile>> waitForExternalField() {
            return Option.of(incomingPatch);
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public boolean isLocallyUploadedFileFinished() {
            return incomingPatch.get().isFinished();
        }

        @Override
        public Option<PostProcessStatus> getPostProcessStatusO() {
            return Option.of(postProcess);
        }

        @Override
        public void xmlize(XmlWriter xw) {
            incomingPatch.xmlize(xw, "incoming-http", Xmlizer2.xmlizeIncomingFileF());
            expectedPatchedMd5.xmlize(xw, "expected-patched-md5", new Function2V<XmlWriter, String>() {
                public void apply(XmlWriter w, String r) {
                    w.addAttribute("md5", r);
                }
            });
            patchedPayloadInfo.xmlize(xw, "patched-file", Xmlizer2.xmlizeContentInfoF());
            Xmlizer2.xmlizePostProcessStatus(xw, postProcess);
        }
    }

    public static class ListArchive extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<UploadedMulcaFile> originalFile;
        @JsonProperty
        public final MutableState<ArchiveListing> listing;

        public ListArchive() {
            this(State.initial(), State.initial(), State.initial());
        }

        @JsonCreator
        public ListArchive(
                @JsonProperty("originalFile") State<UploadedMulcaFile> originalFile,
                @JsonProperty("listing") State<ArchiveListing> listing,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.originalFile = MutableState.cons(originalFile);
            this.listing = MutableState.cons(listing);
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            listing.xmlize(xw, "listing", new Function2V<XmlWriter, ArchiveListing>() {
                public void apply(XmlWriter xw, ArchiveListing result) {
                    Xmlizer2.xmlizeArchiveList(result, xw);
                }
            });
        }
    }

    public static class ExtractArchive extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<ServiceIncomingFile> downloadedFileFromService2;
        @JsonProperty
        public final MutableState<ListF<ExtractedFile>> extractedFiles;
        @JsonProperty
        public final MultipleUpload multipleUpload;

        public ExtractArchive() {
            this(State.initial(), State.initial(), new MultipleUpload(), State.initial());
        }

        @JsonCreator
        public ExtractArchive(
                @JsonProperty("downloadedFileFromService2") @Nullable State<ServiceIncomingFile> downloadedFileFromService2,
                @JsonProperty("extractedFiles") State<ListF<ExtractedFile>> extractedFiles,
                @JsonProperty("multipleUpload") MultipleUpload multipleUpload,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.downloadedFileFromService2 = MutableState.fromNullable(downloadedFileFromService2);
            this.extractedFiles = MutableState.cons(extractedFiles);
            this.multipleUpload = multipleUpload;
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            multipleUpload.xmlize(xw);
        }

    }

    public static class ExtractFileFromArchive extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<ServiceIncomingFile> downloadedFileFromService2;
        @JsonProperty
        public final MutableState<IncomingFile> extractedFile;
        @JsonProperty
        public final MutableState<ContentInfo> payloadInfo;
        @JsonProperty
        public final PostProcessStatus postProcess;

        public ExtractFileFromArchive() {
            this(State.initial(), State.initial(), State.initial(), new PostProcessStatus(), State.initial());
        }

        @JsonCreator
        public ExtractFileFromArchive(
                @JsonProperty("downloadedFileFromService2") @Nullable State<ServiceIncomingFile> downloadedFileFromService2,
                @JsonProperty("extractedFile") State<IncomingFile> extractedFile,
                @JsonProperty("payloadInfo") State<ContentInfo> payloadInfo,
                @JsonProperty("postProcess") PostProcessStatus postProcess,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.downloadedFileFromService2 = MutableState.fromNullable(downloadedFileFromService2);
            this.extractedFile = MutableState.cons(extractedFile);
            this.payloadInfo = MutableState.cons(payloadInfo);
            this.postProcess = postProcess;
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public boolean isLocallyUploadedFileFinished() {
            return downloadedFileFromService2.get().isFinished();
        }

        @Override
        public Option<PostProcessStatus> getPostProcessStatusO() {
            return Option.of(postProcess);
        }

        @Override
        public void xmlize(XmlWriter xw) {
            extractedFile.xmlize(xw, "incoming-http", Xmlizer2.xmlizeIncomingFileF());
            payloadInfo.xmlize(xw, "incoming-file", Xmlizer2.xmlizeContentInfoF());
            Xmlizer2.xmlizePostProcessStatus(xw, postProcess);
        }

    }

    public static class ConvertToMsOoxmlFormat extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<IncomingFile> convertedFile;
        @JsonProperty
        public final MutableState<ContentInfo> payloadInfo;
        @JsonProperty
        public final PostProcessStatus postProcess;

        public ConvertToMsOoxmlFormat() {
            this(State.initial(), State.initial(), new PostProcessStatus(), State.initial());
        }

        @JsonCreator
        public ConvertToMsOoxmlFormat(
                @JsonProperty("convertedFile") State<IncomingFile> convertedFile,
                @JsonProperty("payloadInfo") State<ContentInfo> payloadInfo,
                @JsonProperty("postProcess") PostProcessStatus postProcess,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.convertedFile = MutableState.cons(convertedFile);
            this.payloadInfo = MutableState.cons(payloadInfo);
            this.postProcess = postProcess;
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public boolean isLocallyUploadedFileFinished() {
            return convertedFile.get().isFinished();
        }

        @Override
        public Option<PostProcessStatus> getPostProcessStatusO() {
            return Option.of(postProcess);
        }

        @Override
        public void xmlize(XmlWriter xw) {
            convertedFile.xmlize(xw, "incoming-http", Xmlizer2.xmlizeIncomingFileF());
            payloadInfo.xmlize(xw, "incoming-file", Xmlizer2.xmlizeContentInfoF());
            Xmlizer2.xmlizePostProcessStatus(xw, postProcess);
        }

    }

    public static class GeneratePreview extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<UploadedMulcaFile> originalFile;
        @JsonProperty
        public final MutableState<GenerateImageOnePreviewResult> generatePreview;

        public GeneratePreview() {
            this(State.initial(), State.initial(), State.initial());
        }

        @JsonCreator
        public GeneratePreview(
                @JsonProperty("originalFile") State<UploadedMulcaFile> originalFile,
                @JsonProperty("generatePreview") State<GenerateImageOnePreviewResult> generatePreview,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.originalFile = MutableState.cons(originalFile);
            this.generatePreview = MutableState.cons(generatePreview);
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }

    }

    public static class RegeneratePreview extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<UploadedMulcaFile> originalFile;
        @JsonProperty
        public final PreviewDocumentStatus previewDocumentStatus;
        @JsonProperty
        public final PreviewImageStatus previewImageStatus;
        @JsonProperty
        public final MutableState<FileInformation> videoInfo;
        @JsonProperty
        public final PreviewVideoStatus previewVideoStatus;

        public RegeneratePreview() {
            this(State.initial(), PreviewDocumentStatus.initial(), PreviewImageStatus.initial(), State.initial(),
                    State.initial(), PreviewVideoStatus.initial());
        }

        @JsonCreator
        public RegeneratePreview(
                @JsonProperty("originalFile") State<UploadedMulcaFile> originalFile,
                @JsonProperty("previewDocumentStatus") PreviewDocumentStatus previewDocumentStatus,
                @JsonProperty("previewImageStatus") PreviewImageStatus previewImageStatus,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal,
                @JsonProperty("videoInfo") State<FileInformation> videoInfo,
                @JsonProperty("previewVideoStatus") PreviewVideoStatus previewVideoStatus)
        {
            super(commitFinal);
            this.originalFile = MutableState.cons(originalFile);
            this.previewDocumentStatus = previewDocumentStatus;
            this.previewImageStatus = previewImageStatus;
            this.previewVideoStatus = Option.ofNullable(previewVideoStatus).getOrElse(PreviewVideoStatus.initial());
            this.videoInfo = MutableState.fromNullable(videoInfo);
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }

    }

    public static class ZipFolder extends MpfsRequestStatus {
        public final MutableState<CallbackResponseOption> mpfsFullTree;
        public final MutableState<ParsedTreeResult> parsedFullTree;
        public final MultipleStream multipleStream;

        public Option<ZipArchiveOutputStream> zipOutputStream = Option.empty();
        public final CountDownLatch latch = new CountDownLatch(1);

        public ZipFolder() {
            this(State.initial(), State.initial(), new MultipleStream(), State.initial());
        }

        public ZipFolder(
                State<CallbackResponseOption> mpfsFullTree,
                State<ParsedTreeResult> parsedFullTree,
                MultipleStream multipleStream,
                State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.mpfsFullTree = MutableState.cons(mpfsFullTree);
            this.parsedFullTree = MutableState.cons(parsedFullTree);
            this.multipleStream = multipleStream;
        }

        @Override
        public ListF<String> getPossibleDynamicStages() {
            return Cf.list("ms.s.streamed");
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            // no op
        }

        public boolean isPreparationFailed() {
            return resultWithFinalStages() == UploadRequestStatus.Result.FAILED;
        }

        public boolean isPreparationFinished() {
            return isPreparationFailed() || parsedFullTree.get().getResultO().isPresent();
        }

        public Integer getPreparationErrorCode() {
            Option<Integer> errorCodeO = Option.empty();
            for (MutableState<?> state : Cf.list(mpfsFullTree, parsedFullTree)) {
                if (state.get().isPermanentFailure()) {
                    errorCodeO = PermanentFailureWithCodeException
                            .tryParseStatusCode(state.get().asPermanentFailure().get().getFailureCause().getMessage());
                    break;
                }
            }
            return errorCodeO.getOrElse(500);
        }

    }

    public static class PublishToService extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<UploadedMulcaFile> originalFile2;
        @JsonProperty
        public final MutableState<ServicePublishResult> publishResult;

        public PublishToService() {
            this(State.initial(), State.initial(), State.initial());
        }

        public PublishToService(
                @JsonProperty("originalFile2") State<UploadedMulcaFile> originalFile2,
                @JsonProperty("publishResult") State<ServicePublishResult> publishResult,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.originalFile2 = MutableState.fromNullable(originalFile2);
            this.publishResult = MutableState.cons(publishResult);
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            publishResult.xmlize(xw, "publish", xmlizeServicePublishResult());
        }
    }

    public static class UploadFromService extends MpfsRequestStatus {
        @JsonProperty
        public final MutableState<ServiceIncomingImage> downloadedFileFromService2;
        @JsonProperty
        public final MutableState<ContentInfo> downloadedFileInfo;
        @JsonProperty
        public final PostProcessStatus postProcess;

        public UploadFromService() {
            this(State.initial(), State.initial(), new PostProcessStatus(), State.initial());
        }

        @JsonCreator
        public UploadFromService(
                @JsonProperty("downloadedFileFromService2") @Nullable State<ServiceIncomingImage> downloadedFileFromService2,
                @JsonProperty("downloadedFileInfo") @Nullable State<ContentInfo> downloadedFileInfo,
                @JsonProperty("postProcess") PostProcessStatus postProcess,
                @JsonProperty("commitFinal") @Nullable State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.downloadedFileFromService2 = MutableState.fromNullable(downloadedFileFromService2);
            this.downloadedFileInfo = MutableState.fromNullable(downloadedFileInfo);
            this.postProcess = postProcess;
        }

        @Override
        public boolean isUploadingToDisk() {
            return true;
        }

        @Override
        public boolean isLocallyUploadedFileFinished() {
            return downloadedFileFromService2.get().isFinished();
        }

        @Override
        public Option<PostProcessStatus> getPostProcessStatusO() {
            return Option.of(postProcess);
        }

        @Override
        public void xmlize(XmlWriter xw) {
            downloadedFileInfo.xmlize(xw, "downloaded-file", Xmlizer2.xmlizeContentInfoF());
            Xmlizer2.xmlizePostProcessStatus(xw, postProcess);
        }
    }

    public static class ExportPhotos extends MpfsRequestStatus {
        @JsonProperty
        public final MultiplePhotoExport multiplePhotoExport;

        public ExportPhotos(int numberOfPhotos) {
            this(new MultiplePhotoExport(numberOfPhotos), State.initial());
        }

        @JsonCreator
        public ExportPhotos(
                @JsonProperty("multiplePhotoExport") MultiplePhotoExport multiplePhotoExport,
                @JsonProperty("commitFinal") State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.multiplePhotoExport = multiplePhotoExport;
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
            multiplePhotoExport.xmlize(xw);
        }

    }

    public static class MarkMulcaIdsForRemove extends MpfsRequestStatus {

        @JsonProperty
        public MultipleMarkMulcaIdForRemoveStatus multipleMarkMulcaIdForRemoveStatus;

        @JsonCreator
        public MarkMulcaIdsForRemove(
                @JsonProperty("multipleMarkMulcaIdForRemoveStatus")
                MultipleMarkMulcaIdForRemoveStatus multipleMarkMulcaIdForRemoveStatus,
                @JsonProperty("commitFinal")
                State<CallbackResponseOption> commitFinal)
        {
            super(commitFinal);
            this.multipleMarkMulcaIdForRemoveStatus = multipleMarkMulcaIdForRemoveStatus;
        }

        public MarkMulcaIdsForRemove(int numberOfFiles) {
            this(new MultipleMarkMulcaIdForRemoveStatus(numberOfFiles), State.initial());
        }

        @Override
        public boolean isUploadingToDisk() {
            return false;
        }

        @Override
        public void xmlize(XmlWriter xw) {
        }
    }

    public static Function1B<MpfsRequestStatus> isWaitingForExternalF() {
        return new Function1B<MpfsRequestStatus>() {
            public boolean apply(MpfsRequestStatus mpfsRequestStatus) {
                return mpfsRequestStatus.isWaitingForExternal();
            }
        };
    }

    public static Function1B<MpfsRequestStatus> isUploadingToDiskF() {
        return new Function1B<MpfsRequestStatus>() {
            public boolean apply(MpfsRequestStatus mpfsRequestStatus) {
                return mpfsRequestStatus.isUploadingToDisk();
            }
        };
    }

    public static Function1B<MpfsRequestStatus> isCommitFileUploadFinishedF() {
        return new Function1B<MpfsRequestStatus>() {
            public boolean apply(MpfsRequestStatus mpfsRequestStatus) {
                return mpfsRequestStatus.isCommitFileUploadFinished();
            }
        };
    }

}
