package ru.yandex.chemodan.uploader.services;

import java.io.IOException;

import lombok.AllArgsConstructor;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.commune.uploader.util.http.Content;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.apache.v4.Abstract200ResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class OnlyOfficeApi implements ServiceApi {

    private final String storageHostPort;
    private final String balancerHostPort;
    private final HttpClient httpClient;

    @Override
    public ServicePublishResult publish(ChemodanFile file, Content content) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Content download(ServiceFileId id) {
        HttpUriRequest request = OnlyOfficeRequest.fromServiceFileId(id)
                .map(r -> r.toHttpRequest(balancerHostPort)).getOrElse(() -> getStorageRequest(id));
        return ApacheHttpClientUtils.execute(request, httpClient, new OnlyOfficeResponseHandler());
    }

    private HttpUriRequest getStorageRequest(ServiceFileId id) {
        return new HttpGet(storageHostPort + (id.n.startsWith("/") ? "" : "/") + id.n);
    }

    @Override
    public ChemodanService getService() {
        return ChemodanService.ONLYOFFICE;
    }

    public static class OnlyOfficeContent implements Content {
        private final Option<String> contentType;
        private final byte[] content;

        public OnlyOfficeContent(HttpResponse response) {
            contentType = Cf.x(response.getHeaders("content-type")).firstO().map(Header::getValue);
            try {
                content = EntityUtils.toByteArray(response.getEntity());
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }

        @Override
        public Option<String> getContentType() {
            return contentType;
        }

        @Override
        public InputStreamSource getInputStreamSource() {
            return new ByteArrayInputStreamSource(content);
        }
    }

    private static class OnlyOfficeResponseHandler extends Abstract200ResponseHandler<Content> {
        @Override
        protected Content handle200Response(HttpResponse response) throws ClientProtocolException, IOException {
            return new OnlyOfficeContent(response);
        }
    }
}
