package ru.yandex.chemodan.uploader.services;

import lombok.Data;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.net.uri.Uri2;

/**
 * @author yashunsky
 */

public abstract class OnlyOfficeRequest {
    private static final String ENDPOINT_HOST_KEY = "endpoint_host=";
    private static final String ENDPOINT_SIGN_KEY = "endpoint_sign=";
    private static final String SUBDOMAIN_KEY = "subdomain=";
    private static final String REQUEST_KEY = "request=";

    public static Option<OnlyOfficeRequest> fromServiceFileId(ServiceFileId id) {
        if (id.n.startsWith(ENDPOINT_HOST_KEY)) {
            return Option.of(RequestWithCookie.fromServiceId(id));
        } else if (id.n.startsWith(SUBDOMAIN_KEY)) {
            return Option.of(RequestWithSubdomain.fromServiceId(id));
        } else {
            return Option.empty();
        }
    }

    public abstract HttpUriRequest toHttpRequest(String hostPort);

    private static String trim(String value, String key) {
        Validate.isTrue(value.startsWith(key), "Unable to parse " + value + ". " + key + " prefix expected");
        return value.substring(key.length());
    }

    @Data
    public static class RequestWithCookie extends OnlyOfficeRequest {
        private final String endpointHost;
        private final String endpointSign;
        private final String request;

        public static RequestWithCookie fromServiceId(ServiceFileId id) {
            String[] parts = id.n.split(";", 3);
            if (parts.length != 3) {
                throw new IllegalArgumentException("Unable to parse " + id.toSerializedString());
            }
            return new RequestWithCookie(
                    trim(parts[0], ENDPOINT_HOST_KEY), trim(parts[1], ENDPOINT_SIGN_KEY), trim(parts[2], REQUEST_KEY));
        }

        @Override
        public HttpUriRequest toHttpRequest(String hostPort) {
            String url = hostPort + (request.startsWith("/") ? "" : "/") + request;
            HttpUriRequest httpRequest = new HttpGet(url);
            httpRequest.setHeader("Cookie", ENDPOINT_HOST_KEY + endpointHost + "; " + ENDPOINT_SIGN_KEY + endpointSign);
            return httpRequest;
        }
    }

    @Data
    public static class RequestWithSubdomain extends OnlyOfficeRequest {
        private final String subdomain;
        private final String request;

        public static RequestWithSubdomain fromServiceId(ServiceFileId id) {
            String[] parts = id.n.split(";", 2);
            if (parts.length != 2) {
                throw new IllegalArgumentException("Unable to parse " + id.toSerializedString());
            }
            return new RequestWithSubdomain(trim(parts[0], SUBDOMAIN_KEY), trim(parts[1], REQUEST_KEY));
        }

        @Override
        public HttpUriRequest toHttpRequest(String hostPort) {
            Uri2 uri = Uri2.parse(hostPort);
            String url = uri
                    .withHost(subdomain + "." + uri.getHost().getOrThrow("can not extract host from " + hostPort))
                    .withPath((request.startsWith("/") ? "" : "/") + request).toString();

            return new HttpGet(url);
        }
    }
}
