package ru.yandex.chemodan.uploader.status.strategy;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.util.UploaderContentTypeUtils;
import ru.yandex.chemodan.util.http.ContentTypeUtils;
import ru.yandex.commune.uploader.registry.MutableState;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.util.http.IncomingFile;
import ru.yandex.misc.dataSize.DataSize;

/**
 * @author nshmakov
 */
public class ContentBasedStrategy implements LoadingStatusStrategy {

    @Value("${loadingStatus.coef.default}")
    private int defaultCoef;
    @Value("${loadingStatus.coef.video}")
    private int videoCoef;
    @Value("${loadingStatus.coef.image}")
    private int imageCoef;
    @Value("${loadingStatus.coef.regular}")
    private int regularFileCoef;
    @Value("${loadingStatus.defaultContentLength}")
    private int defaultContentLength;

    @Autowired
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;

    @Override
    public long compute() {
        CollectionF<MpfsRequestRecord> records = uploadRegistry.findRecordsInProgress(Option.empty());
        return records.flatMapO(MpfsRequestRecord.recordToFileStateOF())
                .foldLeft(0L, new Function2<Long, MutableState<IncomingFile>, Long>() {
                    public Long apply(Long acc, MutableState<IncomingFile> state) {
                        for (IncomingFile file : state.get().getIntermediateOrFinalResultO()) {
                            return acc + getCoef(file.getContentType()) * file.getContentLength().getOrElse(
                                    DataSize.fromBytes(defaultContentLength)).toBytes();
                        }
                        return acc;
                    }
                });
    }

    private long getCoef(Option<String> contentTypeO) {
        if (!contentTypeO.isPresent()) {
            return defaultCoef;
        }
        String contentType = contentTypeO.get();
        if (ContentTypeUtils.isVideoContentType(contentType)) {
            return videoCoef;
        } else if (UploaderContentTypeUtils.isImageContentType(contentType)) {
            return imageCoef;
        } else {
            return regularFileCoef;
        }
    }

    @Override
    public LoadingStatusStrategyType getType() {
        return LoadingStatusStrategyType.CONTENT;
    }

    public void setDefaultCoef(int defaultCoef) {
        this.defaultCoef = defaultCoef;
    }

    public void setVideoCoef(int videoCoef) {
        this.videoCoef = videoCoef;
    }

    public void setImageCoef(int imageCoef) {
        this.imageCoef = imageCoef;
    }

    public void setRegularFileCoef(int regularFileCoef) {
        this.regularFileCoef = regularFileCoef;
    }

    public void setDefaultContentLength(int defaultContentLength) {
        this.defaultContentLength = defaultContentLength;
    }

    public void setUploadRegistry(UploadRegistry<MpfsRequestRecord> uploadRegistry) {
        this.uploadRegistry = uploadRegistry;
    }
}
