package ru.yandex.chemodan.uploader.status.strategy;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.uploader.status.strategy.utils.DeviceStatUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.DelayingWorkerThread;
import ru.yandex.misc.worker.Workers;
import ru.yandex.misc.worker.spring.DelayingWorkerServiceBeanSupport;

/**
 * @author nshmakov
 */
public class NetworkIoBasedStrategy extends DelayingWorkerServiceBeanSupport implements LoadingStatusStrategy {

    private static final Logger logger = LoggerFactory.getLogger(NetworkIoBasedStrategy.class);

    private File2 statFile;
    private String deviceName;

    private long lastNetworkIo = 0;
    private volatile long currentNetworkIo = 0;

    @Autowired
    private NetworkIoLoadingStatusDelayHolder delayHolder;

    @PostConstruct
    public void postConstruct() {
        logger.info("NetworkIoBasedStrategy initialized with params: statFile={}, deviceName={}", statFile, deviceName);
    }

    @Override
    protected DelayingWorkerThread worker() {
        return Workers.newDelayingWorker(executable(), workerGroupName, delayHolder.getDelay(), defaultSleepMode(), false);
    }

    @Override
    public LoadingStatusStrategyType getType() {
        return LoadingStatusStrategyType.NETWORK_IO;
    }

    @Override
    public long compute() {
        return currentNetworkIo;
    }

    @Override
    protected void execute() throws Exception {
        refresh();
    }

    public void refresh() {
        String[] deviceStat = DeviceStatUtils.getDeviceStat(deviceName, statFile.readLines()).split(" ");
        long bytesReceived = Long.parseLong(deviceStat[1]);
        long bytesTransmitted = Long.parseLong(deviceStat[9]);
        long currentTotalNetworkIo = bytesReceived + bytesTransmitted;
        if (lastNetworkIo != 0) {
            float denominator = delayHolder.getDelay().getMillis() / 1000f;
            currentNetworkIo = (long) ((currentTotalNetworkIo - lastNetworkIo) / denominator);
        }
        lastNetworkIo = currentTotalNetworkIo;
    }

    public void setStatFile(File2 statFile) {
        this.statFile = statFile;
    }

    public void setNetworkInterface(String deviceName) {
        this.deviceName = deviceName;
    }

    public void setDelayHolder(NetworkIoLoadingStatusDelayHolder delayHolder) {
        this.delayHolder = delayHolder;
    }
}
