package ru.yandex.chemodan.uploader.web;

import java.net.URI;

import javax.servlet.http.HttpServletRequest;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.preview.PreviewImageSize;
import ru.yandex.chemodan.uploader.preview.PreviewSizeParameter;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.web.exception.HttpBadRequestException;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author vavinov
 */
public class ApiArgs {

    public static final String UID = "uid";
    public static final String SPECIAL_UID = "special-uid";
    public static final String FILE_ID = "file-id";
    public static final String PATH = "path";
    public static final String HASH = "hash";
    public static final String ALBUM_PUBLIC_KEY = "album_key";
    public static final String API_VERSION = "api";
    public static final String CALLBACK = "callback";
    public static final String SOURCE_SERVICE = "source-service";
    public static final String SERVICE_FILE_ID = "service-file-id";
    public static final String MULCA_ID = "mulca-id";
    public static final String CONTENT_TYPE = "content-type";
    public static final String ARCHIVE_PATH = "archive-path";
    public static final String ORIGINAL_MD5 = "original-md5";
    public static final String FILE_TO_EXTRACT = "file-to-extract";
    public static final String MAX_FILE_SIZE = "max-file-size";
    public static final String PREVIEW_SIZE = "size";
    public static final String PREVIEW_CROP = "crop";
    public static final String PREVIEW_LOGO_POSITION = "logo-position";
    public static final String PREVIEW_QUALITY = "quality";
    public static final String FILE_SIZE = "file-size";
    public static final String FILE_NAME = "file-name";
    public static final String USE_HTTPS = "use-https";
    public static final String SERVICE_FILE_URL = "service-file-url";
    public static final String LATITUDE = "latitude";
    public static final String LONGITUDE = "longitude";
    public static final String CREATED = "created";
    public static final String PROFILE_ID = "profile-id";
    public static final String ALBUM_ID = "album-id";
    public static final String PREVIEW_STIDS = "pmids";
    public static final String PROVIDER = "provider";
    public static final String PATHS = "paths";
    public static final String DISABLE_RETRIES = "disable-retries";
    public static final String DISABLE_REDIRECTS = "disable-redirects";
    public static final String ALBUM_NAME = "album-name";
    public static final String USER_NAME = "user-name";
    public static final String DYNAMIC_ALBUM_PREVIEW = "dynamic-album-preview";
    public static final String ACCESS_TOKEN = "access_token";
    public static final String ACCESS_TOKEN_TTL = "access_token_ttl";
    public static final String RESOURCE_ID = "resource_id";
    public static final String ZIP_FILES_MPFS_OID = "zip-files-mpfs-oid";
    public static final String EXCLUDE_ORIENTATION = "exclude-orientation";
    public static final String WAIT_COMPLETE_UPLOAD = "wait-complete-upload";
    public static final String WITH_WEBDAV = "with-webdav";
    public static final String WITH_EXIF = "with-exif";
    public static final String WITH_VIRUS_CHECK = "with-virus-check";
    public static final String TLD = "tld";
    public static final String UPLOAD_MAX_SPEED_BPS = "upload-max-speed-bps";

    private static final int MAX_PREVIEW_SIZE = 2 << 12;

    public static ChemodanFile getChemodanFile(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        String fileId = requestX.getRequiredParameter(FILE_ID);
        String path = requestX.getRequiredParameter(PATH);
        return ChemodanFile.cons(getUid(request), fileId, path);
    }

    public static ChemodanFile getChemodanFileWithoutFileId(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        String path = requestX.getRequiredParameter(PATH);
        return ChemodanFile.consWithoutFileId(getUid(request), path);
    }

    public static ListF<ChemodanFile> getChemodanFilesWithoutId(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        String path = requestX.getRequiredParameter(PATHS);
        UidOrSpecial uid = getUid(request);
        return Cf.list(path.split(",")).map(url -> ChemodanFile.consWithoutFileId(uid, url));
    }

    public static UidOrSpecial getUid(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        Option<String> uidArg = r.getParameterO(UID);
        if (uidArg.isPresent()) {
            return UidOrSpecial.uid(PassportUid.cons(Long.parseLong(uidArg.get())));
        } else {
            return UidOrSpecial.special(r.getRequiredParameter(SPECIAL_UID));
        }
    }

    public static Option<PassportUid> getPassportUidO(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        Option<String> uidArg = r.getParameterO(UID);
        if (uidArg.isPresent()) {
            long uid = Long.parseLong(uidArg.get());
            if (uid > 0) {
                return Option.of(PassportUid.cons(uid));
            }
        }
        return Option.empty();
    }

    public static Option<DataSize> getMaxFileSizeO(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        return r.getParameterO(MAX_FILE_SIZE).map(Cf.Long.parseF()).map(DataSize.fromBytesF());
    }

    public static Option<DataSize> getFileSizeO(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        return r.getParameterO(FILE_SIZE).map(Cf.Long.parseF()).map(DataSize.fromBytesF());
    }

    public static Option<Boolean> getUseHttpsO(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        return r.getParameterO(USE_HTTPS).map(Cf.Boolean.parseF());
    }

    public static Option<Boolean> getWaitCompleteUploadO(HttpServletRequest request) {
        HttpServletRequestX r = HttpServletRequestX.wrap(request);
        return r.getParameterO(WAIT_COMPLETE_UPLOAD).map(Cf.Boolean.parseF());
    }

    public static Option<URI> getCallbackUri(HttpServletRequest request) {
        return HttpServletRequestX.wrap(request).getParameterO(ApiArgs.CALLBACK).map(UrlUtils.uriF());
    }

    public static ApiVersion getApiVersion(HttpServletRequest request) {
        return getApiVersionO(request).getOrElse(ApiVersion.V_0_1);
    }

    public static ApiVersion getApiVersionAtLeast(HttpServletRequest request, ApiVersion minimal) {
        Option<ApiVersion> v = getApiVersionO(request);
        String msg = "Must specify " + API_VERSION + " >= " + minimal.toSerializedString();
        validateIsTrue(v.isPresent(), msg);
        validateIsTrue(v.get().ge(minimal), msg);
        return v.get();
    }

    private static Option<ApiVersion> getApiVersionO(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        return requestX.getParameterO(API_VERSION).map(ApiVersion.valueOfF());
    }

    public static Option<String> getTld(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        return requestX.getParameterO(TLD);
    }

    public static Option<Boolean> getDisableRetries(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        return requestX.getParameterO(DISABLE_RETRIES).map(Cf.Boolean.parseF());
    }

    public static Option<Boolean> getDisableRedirects(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        return requestX.getParameterO(DISABLE_REDIRECTS).map(Cf.Boolean.parseF());
    }

    public static Option<Boolean> getDynamicAlbumPreview(HttpServletRequest request) {
        HttpServletRequestX requestX = HttpServletRequestX.wrap(request);
        return requestX.getParameterO(DYNAMIC_ALBUM_PREVIEW).map(Cf.Boolean.parseF());
    }

    public static Option<Boolean> getExcludeOrientation(HttpServletRequestX request) {
        return request.getParameterO(EXCLUDE_ORIENTATION).map(Cf.Boolean.parseF());
    }

    public static String getContentType(HttpServletRequestX request) {
        return request.getParameterO(CONTENT_TYPE).getOrElse("");
    }

    public static String getArchivePath(HttpServletRequestX request) {
        return request.getParameterO(ARCHIVE_PATH).getOrElse("");
    }

    public static boolean getPreviewCrop(HttpServletRequestX request) {
        return request.getParameterO(PREVIEW_SIZE).map(Cf.Boolean.parseF()).getOrElse(false);
    }

    public static Option<Integer> getPreviewQuality(HttpServletRequestX request) {
        Option<Integer> result = request.getNonEmptyParameterO(PREVIEW_QUALITY).map(Cf.Integer.parseF());
        if (result.isPresent()) {
            validateIsTrue(result.get() >= 1 && result.get() <= 100, "Incorrect quality paramter: " + result.get());
        }
        return result;
    }


    public static PreviewSizeParameter getPreviewSize(HttpServletRequestX r) {
        boolean crop = r.getParameterO(ApiArgs.PREVIEW_CROP).map(Cf.Boolean.parseF()).getOrElse(false);
        String sizeParam = r.getRequiredParameter(PREVIEW_SIZE);
        Option<PreviewImageSize> sizeO = PreviewImageSize.R.valueOfO(sizeParam);
        if (sizeO.isPresent()) {
            return new PreviewSizeParameter(
                    PreviewImageSize.R.valueOf(r.getRequiredParameter(PREVIEW_SIZE)));
        } else {
            String width = StringUtils.substringBefore(sizeParam, "x");
            String height = StringUtils.substringAfter(sizeParam, "x");
            validateIsTrue(StringUtils.isNotEmpty(width) || StringUtils.isNotEmpty(height),
                    "Incorrect size parameter");
            int w, h;
            if (StringUtils.isEmpty(width)) {
                h = Integer.parseInt(height);
                w = crop ? h : MAX_PREVIEW_SIZE;
            } else if (StringUtils.isEmpty(height)) {
                w = Integer.parseInt(width);
                h = crop ? w : MAX_PREVIEW_SIZE;
            } else {
                w = Integer.parseInt(width);
                h = Integer.parseInt(height);
            }
            return new PreviewSizeParameter(new Dimension(w, h));
        }
    }

    public static MulcaId getMulcaId(HttpServletRequestX reqX) {
        return MulcaId.fromSerializedString(reqX.getRequiredParameter(ApiArgs.MULCA_ID));
    }

    public static Option<String> getYandexCloudRequestId(HttpServletRequestX reqX) {
        return reqX.getHeaderO(CommonHeaders.YANDEX_CLOUD_REQUEST_ID);
    }

    public static Option<DataSize> getUploadMaxSpeedBps(HttpServletRequestX reqX) {
        try {
            return reqX.getParameterO(UPLOAD_MAX_SPEED_BPS).map(Long::parseLong).map(DataSize::fromBytes);
        } catch (Exception e) {
            return Option.empty();
        }
    }

    public static boolean getBooleanParam(HttpServletRequestX reqX, String name, boolean defaultValue) {
        return reqX.getParameterO(name).map(Boolean::parseBoolean).getOrElse(defaultValue);
    }

    private static void validateIsTrue(boolean condition, String message) {
        if (!condition) {
            throw new HttpBadRequestException(message);
        }
    }
}
