package ru.yandex.chemodan.uploader.web.client;

import java.time.Duration;

import javax.annotation.PreDestroy;

import org.apache.http.HttpHeaders;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.dom4j.Attribute;
import org.dom4j.Element;
import org.dom4j.Node;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.ChemodanService;
import ru.yandex.chemodan.uploader.registry.ApiVersion;
import ru.yandex.chemodan.uploader.services.ServiceFileId;
import ru.yandex.chemodan.uploader.web.ApiArgs;
import ru.yandex.chemodan.uploader.web.ApiUrls;
import ru.yandex.chemodan.util.http.HttpClientUtils;
import ru.yandex.commune.uploader.registry.UploadRequestStatus;
import ru.yandex.commune.uploader.util.http.ContentUtils;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * Only for tests.
 *
 * @author vavinov
 */
public class UploaderClient {
    public static final Logger logger = LoggerFactory.getLogger(UploaderClient.class);

    private final String serviceUrlPrefix;
    private final HttpClient client;

    public interface Urls<T extends BaseStatusXml> {
        String getStatusXmlUrl();
        T requestStatusXml();
        T requestStatusXmlUntilDone();
    }

    public static class BaseStatusXml {
        protected final Element xml;

        protected BaseStatusXml(Element xml) {
            this.xml = xml;
        }

        public final String getStatus() {
            return xml.attributeValue("status");
        }

        public final Element getXml() {
            return xml;
        }
    }

    public class UploadUrls implements Urls<UploadStatusXml> {
        private final Element xml;

        private UploadUrls(Element xml) {
            this.xml = xml;
        }

        public String getPostUrl() {
            return xml.attributeValue("post-target");
        }

        @Override
        public String getStatusXmlUrl() {
            return xml.attributeValue("poll-result");
        }

        /** @return http status code */
        public int upload(InputStreamSource data) {
            return HttpClientUtils.downloadStatus(client,
                    HttpClientUtils.httpPut(getPostUrl(), data.readBytes()));
        }

        /** @return http status code */
        public int upload(InputStreamSource data, MapF<String, String> headers) {
            return HttpClientUtils.downloadStatus(client,
                    HttpClientUtils.httpPut(getPostUrl(), data.readBytes(), headers));
        }

        /** @return http status code */
        public int uploadMultipart(InputStreamSource data) {
            return HttpClientUtils.downloadStatus(client, HttpClientUtils.httpPostMultipart(
                    getPostUrl(), ContentUtils.MULTIPART_FILE_FIELD, data.readText()));
        }

        @Override
        public UploadStatusXml requestStatusXml() {
            HttpGet get = new HttpGet(getStatusXmlUrl());
            // http://jira.codehaus.org/browse/JETTY-1346
            get.addHeader(HttpHeaders.CONNECTION, "Close");
            return new UploadStatusXml(HttpClientUtils.parseXmlResponse(client, get));
        }

        @Override
        public UploadStatusXml requestStatusXmlUntilDone() {
            return requestStatusUntilDone(this);
        }
    }

    public static class UploadStatusXml extends BaseStatusXml {
        private UploadStatusXml(Element xml) {
            super(xml);
        }

        public Option<MulcaId> getFileMulcaId() {
            return attributeValue(xml
                    .selectSingleNode("/upload-info/stages/mulca-file/result/@mulca-id"))
                    .map(MulcaId.fromSerializedStringF());
        }

        public Option<MulcaId> getDigestMulcaId() {
            return attributeValue(xml
                    .selectSingleNode("/upload-info/stages/mulca-digest/result/@mulca-id"))
                    .map(MulcaId.fromSerializedStringF());
        }

        public String getIncomingFileMd5() {
            return attributeValue(xml
                    .selectSingleNode("/upload-info/stages/incoming-file/result/@md5"))
                    .get();
        }

        public Element getGenerateImageOnePreview() {
            return (Element) xml.selectSingleNode("/upload-info/stages/generate-image-one-preview/result");
        }

        public Option<MulcaId> getOnePreviewImageMulcaId() {
            return attributeValue(
                    xml.selectSingleNode("/upload-info/stages/preview-image-mulca-upload/result/@mulca-id"))
                            .map(MulcaId.fromSerializedStringF());
        }

        public Option<MulcaId> getDocumentPreviewMulcaId() {
            return attributeValue(
                    xml.selectSingleNode("/upload-info/stages/preview/result/@mulca-id"))
                            .map(MulcaId.fromSerializedStringF());
        }

        public String getExifCreationDate() {
            return attributeValue(xml.selectSingleNode("/upload-info/stages/exif-info/result/@creation-date")).get();
        }

        public String getExifLatitude() {
            return attributeValue(xml.selectSingleNode("/upload-info/stages/exif-info/result/@latitude")).get();
        }

        public String getMediaCreationDate() {
            return attributeValue(xml.selectSingleNode("/upload-info/stages/media-info/result/@creation-date")).get();
        }

        public Option<MulcaId> getVideoPreviewMulcaId() {
            return attributeValue(xml.selectSingleNode("/upload-info/stages/preview-video-mulca-upload/result/@mulca-id"))
                    .map(MulcaId.fromSerializedStringF());
        }

        public Option<String> getVideoInfo() {
            Element element = (Element) xml.selectSingleNode("/upload-info/stages/video-info/result");
            return element != null ? Option.ofNullable(element.getText()) : Option.empty();
        }

        public String getExifLongitude() {
            return attributeValue(xml.selectSingleNode("/upload-info/stages/exif-info/result/@longitude")).get();
        }

    }

    public UploaderClient(String serviceUrlPrefix, Timeout timeout) {
        this.serviceUrlPrefix = serviceUrlPrefix;
        this.client = ApacheHttpClientUtils.multiThreadedHttpsClient(
                timeout, 1, "uploader-client-for-tests");
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(client);
    }

    public UploadUrls uploadToDisk(ChemodanFile file) {
        String uri = serviceUrlPrefix + ApiUrls.UPLOAD_URL + "/" +
                ChemodanService.DISK.name().toLowerCase();
        return new UploadUrls(HttpClientUtils.parseXmlResponse(client, HttpClientUtils.httpPost(
                uri, toParameterMap(file).plus1(
                        ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString()))));
    }

    public UploadStatusXml uploadToDiskUntilDone(ChemodanFile file, InputStreamSource data) {
        UploadUrls urls = uploadToDisk(file);
        int code = urls.upload(data);
        Check.C.isTrue(HttpStatus.is2xx(code), "Status is not 2xx! ", code);
        return requestStatusUntilDone(urls);
    }

    public class PublishUrls implements Urls<PublishStatusXml> {
        private final Element xml;

        private PublishUrls(Element xml) {
            this.xml = xml;
        }

        @Override
        public PublishStatusXml requestStatusXml() {
            return new PublishStatusXml(
                    HttpClientUtils.parseXmlResponse(client, new HttpGet(getStatusXmlUrl())));
        }

        @Override
        public String getStatusXmlUrl() {
            return xml.attributeValue("poll-result");
        }

        @Override
        public PublishStatusXml requestStatusXmlUntilDone() {
            return requestStatusUntilDone(this);
        }
    }

    public static class PublishStatusXml extends BaseStatusXml {
        private PublishStatusXml(Element xml) {
            super(xml);
        }

        public ServiceFileId getServiceFileId() {
            return ServiceFileId.valueOf(attributeValue(
                xml.selectSingleNode("/publish-info/stages/publish/result/@service-file-id"))
                .getOrThrow(xml.asXML()));
        }
    }

    public PublishUrls publishToService(ChemodanFile file, MulcaId mulcaId,
            ChemodanService targetService)
    {
        String uri = serviceUrlPrefix + ApiUrls.PUBLISH + "/" + targetService.name().toLowerCase();
        return new PublishUrls(HttpClientUtils.parseXmlResponse(client, HttpClientUtils.httpPost(
                uri, toParameterMap(file)
                        .plus1(ApiArgs.MULCA_ID, mulcaId.getStidCheckNoPart())
                        .plus1(ApiArgs.API_VERSION, ApiVersion.V_0_2.toSerializedString()))));
    }

    public static class UploadFromServiceStatusXml extends BaseStatusXml {
        public UploadFromServiceStatusXml(Element xml) {
            super(xml);
        }

        public Option<MulcaId> getFileMulcaId() {
            return attributeValue(xml
                    .selectSingleNode("/upload-from-service/stages/mulca-file/result/@mulca-id"))
                    .map(MulcaId.fromSerializedStringF());
        }
    }

    ////

    public static MapF<String, Object> toParameterMap(ChemodanFile file) {
        return Cf.<String, Object>map(ApiArgs.UID, file.getPassportUid().getUid())
                .plus1(ApiArgs.FILE_ID, file.getUniqueFileId())
                .plus1(ApiArgs.PATH, file.getPath());
    }

    public static MapF<String, Object> toParameterMapWithoutFileId(ChemodanFile file) {
        return Cf.<String, Object>map(ApiArgs.UID, file.getPassportUid().getUid())
                .plus1(ApiArgs.PATH, file.getPath());
    }

    private static Option<String> attributeValue(Node a) {
        return a == null ? Option.empty() : Option.of(((Attribute) a).getValue());
    }

    private static <S extends BaseStatusXml> S requestStatusUntilDone(Urls<S> urls) {
        Duration limitTimeout = Duration.ofSeconds(5);
        Duration sleepTimeout = Duration.ofMillis(200);
        long maxCount = limitTimeout.toMillis() / sleepTimeout.toMillis();
        for (int i = 0; i < maxCount; i++) {
            S status = urls.requestStatusXml();
            if (status.getStatus().equals(UploadRequestStatus.Result.COMPLETED.name().toLowerCase())) {
                return status;
            }
            Check.C.notEquals(UploadRequestStatus.Result.FAILED.name().toLowerCase(), status.getStatus());
            ThreadUtils.sleep(200);
        }

        S status = urls.requestStatusXml();
        throw new IllegalStateException("Timeout in task processing, status: "
                + status.getStatus() + ", info: " + status.getXml().asXML());
    }
}

