package ru.yandex.chemodan.uploader.web.control;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.uploader.mulca.MulcaUploadInfo;
import ru.yandex.chemodan.uploader.registry.RequestStatesHandler;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord.RegeneratePreview;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecordUtils;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.Xmlizer2;
import ru.yandex.chemodan.uploader.registry.record.status.GenerateImageOnePreviewResult;
import ru.yandex.chemodan.uploader.registry.record.status.MpfsRequestStatus;
import ru.yandex.chemodan.uploader.web.ApiArgs;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.StageListenerPool;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.commune.video.format.FileInformation;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.xml.stream.JavolutionXmlWriter;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * @author akirakozov
 */
@SuppressWarnings("serial")
public class RegeneratePreviewServlet extends HttpServlet {

    private static final Logger logger = LoggerFactory.getLogger(RegeneratePreviewServlet.class);

    @Value("${regenerate.preview.action.timeout:-5m}")
    private Duration timeout;

    @Autowired
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;

    @Autowired
    private RequestStatesHandler requestStatesHandler;

    @Autowired
    private StageListenerPool<Record<?>> stageListenerPool;

    private MpfsRequestRecord createRequestRecord(HttpServletRequest req) {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);
        return uploadRegistry.createRecord(MpfsRequestRecordUtils.consF(
                new MpfsRequest.RegeneratePreview(
                        ApiArgs.getApiVersion(req),
                        ApiArgs.getMulcaId(reqX),
                        ApiArgs.getContentType(reqX),
                        ApiArgs.getFileSizeO(req),
                        reqX.getParameterO(ApiArgs.FILE_NAME),
                        ApiArgs.getYandexCloudRequestId(reqX)),
                RequestRevision.initial(HostInstant.hereAndNow())));
    }

    private static Option<MulcaId> getPreviewMulcaId(MpfsRequestStatus.RegeneratePreview status) {
        for (MulcaUploadInfo info : status.previewDocumentStatus.previewMulcaUploadInfo.get().getResultO()) {
            return Option.of(info.getMulcaId());
        }
        for (MulcaUploadInfo info : status.previewImageStatus.previewMulcaUploadInfo.get().getResultO()) {
            return Option.of(info.getMulcaId());
        }
        for (MulcaUploadInfo info : status.previewVideoStatus.previewMulcaUploadInfo.get().getResultO()) {
            return Option.of(info.getMulcaId());
        }
        return Option.empty();
    }

    private static Option<MulcaId> getMultiplePreviewMulcaId(MpfsRequestStatus.RegeneratePreview status) {
        for (MulcaUploadInfo info : status.previewVideoStatus.multiplePreviewMulcaUploadInfo.get().getResultO()) {
            return Option.of(info.getMulcaId());
        }
        return Option.empty();
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        MpfsRequestRecord record = createRequestRecord(req);

        Ndc.push(record.meta.id.toString());
        processRequest(resp, record);
    }

    private void processRequest(final HttpServletResponse resp, MpfsRequestRecord record) {
        final CountDownLatch latch = new CountDownLatch(1);
        final AtomicInteger result = new AtomicInteger(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
        final AtomicBoolean shouldWriteResponse = new AtomicBoolean(true);

        stageListenerPool.addStageListener(record.meta.id, new Function1V<Record<?>>() {
            public void apply(Record<?> r) {
                MpfsRequestStatus.RegeneratePreview status =
                        (MpfsRequestStatus.RegeneratePreview) r.get().getStatus();

                if (status.isFinishedWithoutFinalStages()) {
                    try {
                        Option<MulcaId> previewMulcaId = getPreviewMulcaId(status);
                        if (previewMulcaId.isPresent()) {
                            if (shouldWriteResponse.getAndSet(false)) {
                                writeResponse(resp, previewMulcaId.get(), status.videoInfo.get().getResultO(),
                                        getMultiplePreviewMulcaId(status),
                                        status.previewImageStatus.generateOnePreview.get().getResultO());
                            }
                            result.set(HttpStatus.SC_200_OK);
                        } else if (status.originalFile.get().isFailure()) {
                            result.set(HttpStatus.SC_404_NOT_FOUND);
                        } else {
                            State<File2> documentState = status.previewDocumentStatus.generatePreview.get();
                            State<GenerateImageOnePreviewResult> imageState =
                                    status.previewImageStatus.generateOnePreview.get();
                            State<FileInformation> videoInfoStatus = status.videoInfo.get();
                            State<File2> previewVideoStatus = status.previewVideoStatus.generatePreview.get();

                            if (imageState.isFailure() || documentState.isFailure()
                                    || videoInfoStatus.isFailure() || previewVideoStatus.isFailure()
                                || (imageState.isSkipped() && documentState.isSkipped()
                                    && videoInfoStatus.isSkipped() && previewVideoStatus.isSkipped()))
                            {
                                result.set(HttpStatus.SC_400_BAD_REQUEST);
                            }
                        }
                    } catch (Exception e) {
                        logger.warn(e, e);
                    } finally {
                        logger.debug("Release lock for record: " + r.get().meta.id);
                        latch.countDown();
                    }
                }
            }
        });

        boolean wasTimeout;
        try {
            uploadRegistry.saveRecord(record);
            wasTimeout = !ThreadUtils.await(latch, timeout);
        } catch (RuntimeException ex) {
            logger.debug("Got error while saving record and waiting for preview regeneration", ex);
            throw ex;
        } finally {
            stageListenerPool.removeStageListener(record.meta.id);
        }

        if (wasTimeout) {
            logger.warn("Got timeout while waiting for preview generation");

            resp.setStatus(HttpStatus.SC_408_REQUEST_TIMEOUT);

            Record<RegeneratePreview> currentRecord =
                    Record.cons(uploadRegistry, (RegeneratePreview) uploadRegistry.findRecord(record.meta.id).get());
            requestStatesHandler.handleInternalError(currentRecord,
                    new RuntimeException("Servlet request timeout: " + timeout));
        } else {
            logger.debug("Completed preview regeneration with status#{}", result.get());
            resp.setStatus(result.get());
        }
    }

    // TODO: extract result xml writer to separate class
    private void writeResponse(HttpServletResponse resp, MulcaId mulcaId, Option<FileInformation> fileInformation,
            Option<MulcaId> multiplePreviewMuldaId, Option<GenerateImageOnePreviewResult> imageResult)
            throws IOException
    {
        XmlWriter xw = new JavolutionXmlWriter(resp.getOutputStream(), false, "\t");
        resp.setContentType("application/xml");

        xw.startElement("regenerate-preview");
        xw.textElement("mulca-id", mulcaId.getStidCheckNoPart());
        if (multiplePreviewMuldaId.isPresent()) {
            xw.textElement("multiple-preview-mulca-id", multiplePreviewMuldaId.get().getStidCheckNoPart());
        }
        if (fileInformation.isPresent()) {
            byte[] serializedJson = FileInformation.PS.getSerializer().serializeJson(fileInformation.get());
            xw.textElement("video-info", new String(serializedJson, CharsetUtils.UTF8_CHARSET));
        }

        if (imageResult.isPresent()) {
            xw.startElement("image-info");
            Xmlizer2.generateOneImagePreviewResultF().apply(xw, imageResult.get());
            xw.endElement();
        }

        xw.endElement();
        xw.flush();
        xw.close();
    }
}
