package ru.yandex.chemodan.uploader.web.control;

import java.io.IOException;
import java.nio.charset.Charset;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.uadetector.OperatingSystemFamily;
import net.sf.uadetector.UserAgent;
import net.sf.uadetector.UserAgentStringParser;
import net.sf.uadetector.service.UADetectorServiceFactory;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.mpfs.MpfsAlbumKey;
import ru.yandex.chemodan.mpfs.MpfsFilesKey;
import ru.yandex.chemodan.uploader.ChemodanFile;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequest;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord.ZipFolder;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecordUtils;
import ru.yandex.chemodan.uploader.web.ApiArgs;
import ru.yandex.commune.uploader.registry.RequestRevision;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.util.HostInstant;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.concurrent.CountDownLatches;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.web.servlet.HttpRequestUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * TODO learn how to create zip archives with utf-8 entries
 * to get rid of character detection necessify afterwards
 *
 * @author ssytnik
 */
@SuppressWarnings("serial")
public class ZipFolderServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(ZipFolderServlet.class);

    private static UserAgentStringParser UA_PARSER = UADetectorServiceFactory.getResourceModuleParser();

    private static Charset CP857_TR_CHARSET = Charset.forName("cp857");
    private static Charset CP866_RU_CHARSET = Charset.forName("cp866");

    @Value("${zip.folder.action.timeout}")
    private Duration timeout;

    // 0 = no compression, 1 = best speed ... 9 = best compression
    @Value("${zip.folder.compression}")
    private int compressionLevel;

    @Autowired
    private UploadRegistry<ZipFolder> uploadRegistry;

    public enum Type {
        PUBLIC_FOLDER, PRIVATE_FOLDER, ALBUM, FILES
    }

    private final Type type;

    public ZipFolderServlet(Type type) {
        this.type = type;
    }

    private ZipFolder createRequestRecord(HttpServletRequest req) {
        HttpServletRequestX reqX = HttpServletRequestX.wrap(req);

        Option<MpfsAlbumKey> albumKey = Option.empty();
        Option<String> hash = Option.empty();
        Option<MpfsFilesKey> filesKey = Option.empty();
        ChemodanFile chemodanFile = ChemodanFile.FAKE;

        switch (type) {
            case ALBUM:
                String publicKey = reqX.getRequiredParameter(ApiArgs.ALBUM_PUBLIC_KEY);
                Option<PassportUid> uidO = ApiArgs.getPassportUidO(req);
                albumKey = Option.of(new MpfsAlbumKey(publicKey, uidO));
                break;
            case PUBLIC_FOLDER:
                hash = Option.of(reqX.getRequiredParameter(ApiArgs.HASH));
                break;
            case PRIVATE_FOLDER:
                chemodanFile = ApiArgs.getChemodanFileWithoutFileId(req);
                break;
            case FILES:
                String mpfsOid = reqX.getRequiredParameter(ApiArgs.ZIP_FILES_MPFS_OID);
                PassportUid uid = ApiArgs.getUid(req).toPassportUidOrAnonymous().toPassportUid();
                filesKey = Option.of(new MpfsFilesKey(mpfsOid, uid));
                break;
        }

        return uploadRegistry.createRecord(MpfsRequestRecordUtils.consF(
                new MpfsRequest.ZipFolder(
                        ApiArgs.getApiVersion(req), chemodanFile, hash, albumKey, filesKey,
                        ApiArgs.getYandexCloudRequestId(reqX)),
                RequestRevision.initial(HostInstant.hereAndNow())));
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        ZipFolder record = createRequestRecord(req);
        record.getStatus().zipOutputStream = Option.of(prepareZipOutputStream(req, resp));
        Ndc.push(record.meta.id.toString());
        uploadRegistry.saveRecord(record);
        if (!CountDownLatches.await(record.getStatus().latch, timeout)) {
            logger.warn("Timeout of {} reached -- aborting", timeout);
            throw new HttpException(HttpStatus.SC_408_REQUEST_TIMEOUT);
        }
        if (record.getStatus().isPreparationFailed()) {
            resp.setStatus(record.getStatus().getPreparationErrorCode());
        }
    }

    private ZipArchiveOutputStream prepareZipOutputStream(HttpServletRequest request,
            HttpServletResponse response) throws IOException
    {
        response.setContentType("application/zip");

        ZipArchiveOutputStream zipOutputStream = new ZipArchiveOutputStream(response.getOutputStream());
        zipOutputStream.setMethod(ZipArchiveOutputStream.DEFLATED);
        zipOutputStream.setLevel(compressionLevel);
        String nativeZipEncoding = determineNativeZipEncoding(request);
        zipOutputStream.setEncoding(nativeZipEncoding);
        zipOutputStream.setCreateUnicodeExtraFields(ZipArchiveOutputStream.UnicodeExtraFieldPolicy.ALWAYS);
        zipOutputStream.setFallbackToUTF8(false);

        return zipOutputStream;
    }

    private String determineNativeZipEncoding(HttpServletRequest request) {
        Charset res = CharsetUtils.UTF8_CHARSET;

        Option<String> uaHeader = HttpRequestUtils.getUserAgent(request);
        if (uaHeader.isPresent()) {
            UserAgent agent = UA_PARSER.parse(uaHeader.get());
            OperatingSystemFamily osFamily = agent.getOperatingSystem().getFamily();
            String language = request.getLocale().getLanguage();
            if (osFamily == OperatingSystemFamily.WINDOWS) {
                if ("tr".equals(language)) {
                    res = CP857_TR_CHARSET;
                } else {
                    res = CP866_RU_CHARSET;
                }
            }
            logger.debug("Chosen '{}' for OS '{}' and language '{}'",
                    res.name(), osFamily.getName(), language);
        }

        return res.name();
    }
}
