package ru.yandex.chemodan.uploader.web.control.sync;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.uploader.processor.SyncRequestProcessor;
import ru.yandex.chemodan.uploader.registry.MulcaDownloadException;
import ru.yandex.chemodan.uploader.web.exception.HttpErrorException;
import ru.yandex.chemodan.util.exception.PermanentHttpFailureException;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author akirakozov
 */
@SuppressWarnings("serial")
public abstract class AbstractSyncRequestServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(AbstractSyncRequestServlet.class);

    @Value("${sync.request.default.timeout:-1m}")
    private Duration timeout;

    @Autowired
    protected SyncRequestProcessor processor;

    @Override
    protected void doGet(HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {
        final HttpServletRequestX reqX = HttpServletRequestX.wrap(req);

        try {
            ThreadLocalTimeout.withTimeout(getTimeout(), new Function0V() {
                public void apply() {
                    try {
                        handleRequest(reqX, resp);
                    } catch (IOException e) {
                        throw IoUtils.translate(e);
                    }
                }
            });
        } catch (PermanentHttpFailureException e) {
            if (e.getStatusCode().isSome(HttpStatus.SC_404_NOT_FOUND)) {
                resp.setStatus(HttpStatus.SC_404_NOT_FOUND);
            } else {
                resp.setStatus(HttpStatus.SC_503_SERVICE_UNAVAILABLE);
            }
        } catch (MulcaDownloadException e) {
            resp.setStatus(HttpStatus.SC_503_SERVICE_UNAVAILABLE);
        } catch (ThreadLocalTimeoutException e) {
            resp.setStatus(HttpStatus.SC_408_REQUEST_TIMEOUT);
        } catch (HttpErrorException e) {
            logger.warn(e, e);
            resp.setStatus(e.getHttpErrorCode());
        }
    }

    protected abstract void handleRequest(HttpServletRequestX reqX, HttpServletResponse resp) throws IOException;

    protected Duration getTimeout() {
        return timeout;
    }

    public void setTimeout(Duration timeout) {
        this.timeout = timeout;
    }

    public void setProcessor(SyncRequestProcessor processor) {
        this.processor = processor;
    }
}
