package ru.yandex.chemodan.uploader.web.data;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicReference;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.uploader.registry.StageListenerPoolWithChildren;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.web.UploadTimeoutHolder;
import ru.yandex.chemodan.uploader.web.control.UploadUrlServlet;
import ru.yandex.chemodan.uploader.web.data.util.UploadServletUtils;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.local.queue.LocalQueueProcessor;
import ru.yandex.commune.uploader.local.queue.LocalQueuePush;
import ru.yandex.commune.uploader.registry.StageUtils;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestStatus;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.web.data.DiskWritePolicy;
import ru.yandex.commune.uploader.web.data.RangedPutUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.web.servlet.HttpRequestUtils;

/**
 * For load testing.
 *
 * @author vavinov
 */
public class SyncUploadServlet extends DataHttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(SyncUploadServlet.class);

    @Autowired
    private UploadUrlServlet uploadUrlServlet;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;
    @Autowired
    private LocalQueueProcessor localQueueProcessor;
    @Autowired
    private LocalQueuePush localQueuePush;
    @Autowired
    private DiskWritePolicy diskWritePolicy;
    @Autowired
    private StageListenerPoolWithChildren<Record<?>> stageListenerPool;
    @Autowired
    private UploadTimeoutHolder uploadTimeoutHolder;

    @Override
    protected void doPut(final HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        final long startNanoseconds = System.nanoTime();

        final Record<MpfsRequestRecord.UploadToDefault> record = Record.cons(
                uploadRegistry, (MpfsRequestRecord.UploadToDefault) uploadUrlServlet.saveRecord(req));

        final CountDownLatch latch = new CountDownLatch(1);
        final AtomicReference<UploadRequestStatus.Result> result = new AtomicReference<UploadRequestStatus.Result>();
        localQueueProcessor.setListener(record.get().meta.id, new Function1V<UploadRequestStatus.Result>() {
            public void apply(UploadRequestStatus.Result r) {
                result.set(r);
                latch.countDown();
            }
        });

        try {
            Function0<PutResult> processUploadF = () -> RangedPutUtils.processPut(localFileManager, record,
                    RangedPutUtils.makePutRequestContext(req), diskWritePolicy);

             PutResult putResult = UploadServletUtils.handleAndExecuteAsap(
                    record.get(), localQueuePush, stageListenerPool, uploadTimeoutHolder.getUploadTimeout(req),
                    processUploadF, HttpRequestUtils.getUserAgent(req));

            Check.equals(PutResult.COMPLETED, putResult);
            final long afterUploadNanoseconds = System.nanoTime();

            ThreadUtils.await(latch);

            final long nowNanoseconds = System.nanoTime();
            logger.debug("{} {} time={}ms = {}ms + {}ms", result.get(), record.get().meta.id,
                    (nowNanoseconds - startNanoseconds) / 1000000L,
                    (afterUploadNanoseconds - startNanoseconds) / 1000000L,
                    (nowNanoseconds - afterUploadNanoseconds) / 1000000L);
            if (result.get() == UploadRequestStatus.Result.COMPLETED) {
                resp.setStatus(HttpStatus.SC_200_OK);
            } else if (result.get() == UploadRequestStatus.Result.FAILED) {
                resp.sendError(record.get().getStatus()
                        .getFailureStatusCode()
                        .filter(StageUtils::isFirstCallbackFailure)
                        .orElse(HttpStatus.SC_503_SERVICE_UNAVAILABLE));
            } else {
                resp.sendError(HttpStatus.SC_500_INTERNAL_SERVER_ERROR);
            }
        } finally {
            localQueueProcessor.removeListener(record.get().meta.id);
        }
    }
}
