package ru.yandex.chemodan.uploader.web.data;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.uploader.registry.StageListenerPoolWithChildren;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.web.UploadTimeoutHolder;
import ru.yandex.chemodan.uploader.web.data.util.UploadServletUtils;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.local.queue.LocalQueuePush;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.web.data.DiskWritePolicy;
import ru.yandex.commune.uploader.web.data.RangedPutUtils;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.web.servlet.HttpRequestUtils;

/**
 * @author vavinov
 */
public class UploadDeltaTargetServlet extends DataHttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(UploadDeltaTargetServlet.class);

    @Autowired
    private UploadRegistry<MpfsRequestRecord> uploadRegistry;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    private LocalQueuePush localQueuePush;
    @Autowired
    private DiskWritePolicy diskWritePolicy;
    @Autowired
    private StageListenerPoolWithChildren<Record<?>> stageListenerPool;
    @Autowired
    private UploadTimeoutHolder uploadTimeoutHolder;

    private MpfsRequestRecord.PatchAtDefault getRecord(HttpServletRequest req) {
        logger.info(HttpRequestUtils.describeRemoveSensitive(req));
        final UploadRequestId id = UploadRequestId.valueOf(HttpRequestUtils.localUri(req));
        return uploadRegistry.findRecordAssignedToLocalhost(id, MpfsRequestRecord.PatchAtDefault.class);
    }

    @Override
    public void doHead(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException
    {
        final MpfsRequestRecord.PatchAtDefault record = getRecord(req);

        Ndc.Handle h = Ndc.push(record.meta.id.toString());
        try {
            RangedPutUtils.processHeadAndSendResponse(record.getStatus().incomingPatch.get(), resp);
        } finally {
            h.popSafely();
        }
    }

    @Override
    public void doPut(final HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException
    {
        final Record<MpfsRequestRecord.PatchAtDefault> record =
                Record.cons(uploadRegistry, getRecord(req));

        PutResult result = UploadServletUtils.handleAndExecuteAsap(
            record.get(), localQueuePush, stageListenerPool, uploadTimeoutHolder.getUploadTimeout(req),
            new Function0<PutResult>() {
                public PutResult apply() {
                    Validate.V.equals(record.get().getRequest().originalMd5, req.getHeader("If-Match"),
                            "Original file md5 does not match If-Match: header.");

                    final String yandexDiffHeader = Option.ofNullable(req.getHeader("Yandex-Diff"))
                            .getOrThrow("Expected Yandex-Diff: header");
                    State<String> expectedPatchedMd5 = record.get().getStatus().expectedPatchedMd5.get();
                    if (expectedPatchedMd5.isSuccess()) {
                        Validate.V.equals(yandexDiffHeader, expectedPatchedMd5.getResultO().get(),
                                "Yandex-Diff: header value changed during upload.  Request a different /patch-url.");
                    } else {
                        logger.info("Getting expected patched md5 from Yandex-Diff: {}", yandexDiffHeader);
                        record.update(record.get().getStatus().expectedPatchedMd5,
                                expectedPatchedMd5.asInitial().get().completeInstantly(new Instant(), yandexDiffHeader));
                    }

                    return RangedPutUtils.processPut(localFileManager, record,
                            RangedPutUtils.makePutRequestContext(req), diskWritePolicy);
                }
            },
            HttpRequestUtils.getUserAgent(req));

        int statusCode = (result == PutResult.FAILED) ? HttpStatus.SC_409_CONFLICT : result.getStatusCode(); // https://st.yandex-team.ru/CHEMODAN-16174
        resp.setStatus(statusCode);
    }
}
