package ru.yandex.chemodan.uploader.web.data;

import java.io.IOException;
import java.util.function.Consumer;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.http.CommonHeaders;
import ru.yandex.chemodan.log.DiskLog4jRequestLog;
import ru.yandex.chemodan.uploader.UidOrSpecial;
import ru.yandex.chemodan.uploader.mulca.SimultaneousMulcaUploadManager;
import ru.yandex.chemodan.uploader.registry.StageListenerPoolWithChildren;
import ru.yandex.chemodan.uploader.registry.record.Digests;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.web.UploadTimeoutHolder;
import ru.yandex.chemodan.uploader.web.YaDiskUserAgent;
import ru.yandex.chemodan.uploader.web.data.util.CancelableUploadInputStreamX;
import ru.yandex.chemodan.uploader.web.data.util.OnTheFlyDigester;
import ru.yandex.chemodan.uploader.web.data.util.UploadServletUtils;
import ru.yandex.chemodan.uploader.web.data.util.UserUploadSpeedLimitManager;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.uploader.local.file.LocalFileManager;
import ru.yandex.commune.uploader.local.queue.LocalQueuePush;
import ru.yandex.commune.uploader.registry.RecordNotFoundException;
import ru.yandex.commune.uploader.registry.RecordWrapper;
import ru.yandex.commune.uploader.registry.State;
import ru.yandex.commune.uploader.registry.UploadRegistry;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.http.MultipartUtils;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.web.data.DiskWritePolicy;
import ru.yandex.commune.uploader.web.data.HttpPutRequestContext;
import ru.yandex.commune.uploader.web.data.HttpPutRequestContextUtils;
import ru.yandex.commune.uploader.web.data.RangedPutUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bytes.ByteArrayByteSequence;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.web.servlet.HttpRequestUtils;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * http://wiki.yandex-team.ru/Pochta/chemodan/uploader/api
 *
 * @author vavinov
 */
public class UploadTargetServlet extends DataHttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(UploadTargetServlet.class);

    @Autowired
    private UploadRegistry<MpfsRequestRecord.UploadToDefault> uploadRegistry;
    @Autowired
    private LocalFileManager localFileManager;
    @Autowired
    private LocalQueuePush localQueuePush;
    @Autowired
    private DiskWritePolicy diskWritePolicy;
    @Autowired
    private StageListenerPoolWithChildren<Record<?>> stageListenerPool;
    @Autowired
    private UploadTimeoutHolder uploadTimeoutHolder;
    @Autowired
    private SimultaneousMulcaUploadManager simultaneousMulcaUploadManager;
    @Autowired
    private UserUploadSpeedLimitManager userUploadSpeedLimitManager;

    private final DynamicProperty<Option<Integer>> previouslyFailedUploadsStatusCode;

    public UploadTargetServlet() {
        this.previouslyFailedUploadsStatusCode = DynamicProperty.cons("uploader-failed-uploads-status-code",
                Option.of(HttpStatus.SC_500_INTERNAL_SERVER_ERROR));
    }

    static String urlEncodeDocumentPath(String fullPath) {
        // Location should be encoded with percent-encoding (CHEMODAN-3657)
        String path = StringUtils.substringAfter(fullPath, ":");
        return StringUtils.join(
                    Cf.x(StringUtils.split(path, "/"))
                    .map(UrlUtils.urlEncodeF())
                    .map(StringUtils.addPrefixF("/"))
               , "").replace("+", "%20");
    }

    @Override
    public void doOptions(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException
    {
        super.doOptions(req, resp);
    }

    @Override
    public void doHead(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException
    {
        MpfsRequestRecord.UploadToDefault record = getRecordAndLogRequest(req);

        Ndc.Handle h = Ndc.push(record.meta.id.toString());
        try {
            RangedPutUtils.processHeadAndSendResponse(record.getStatus().userFile.get(), resp);
        } finally {
            h.popSafely();
        }
    }

    @Override
    public void doPost(final HttpServletRequest req, HttpServletResponse resp) {
        if (!ServletFileUpload.isMultipartContent(req)) {
            logger.debug("Files may be POSTed only as multipart/form-data.");
            resp.setStatus(HttpStatus.SC_400_BAD_REQUEST);
        } else {
            final Record<MpfsRequestRecord.UploadToDefault> record =
                    Record.cons(uploadRegistry, getRecordAndLogRequest(req));

            if (!record.get().getStatus().userFile.get().asInitial().isPresent()) {
                logger.debug("POST/multipart upload cannot be resumed.");
                resp.setStatus(HttpStatus.SC_409_CONFLICT);
            } else {
                checkAndPerformUpload(record, req, resp);
            }
        }
    }

    @Override
    public void doPut(final HttpServletRequest req, HttpServletResponse resp) {
        final Record<MpfsRequestRecord.UploadToDefault> record = Record.cons(uploadRegistry, getRecordAndLogRequest(req));

        String fullPath = record.get().getRequest().chemodanFile.getPath();
        resp.setHeader(CommonHeaders.LOCATION, urlEncodeDocumentPath(fullPath));

        checkAndPerformUpload(record, req, resp);
    }

    /**
     * Checks if user upload permanently failed previously and performs upload if it didn't
     */
    private void checkAndPerformUpload(Record<MpfsRequestRecord.UploadToDefault> record,
            HttpServletRequest request, HttpServletResponse response)
    {
        response.setStatus(getStatusCodeIfFailedPreviously(record).getOrElse(
                () -> performUpload(record, request).getStatusCode()));
    }

    private PutResult performUpload(Record<MpfsRequestRecord.UploadToDefault> record, HttpServletRequest request) {
        return UploadServletUtils.handleAndExecuteAsap(record.get(), localQueuePush, stageListenerPool,
                uploadTimeoutHolder.getUploadTimeout(request), processUploadF(record, request),
                HttpRequestUtils.getUserAgent(request));
    }

    private Option<Integer> getStatusCodeIfFailedPreviously(Record<MpfsRequestRecord.UploadToDefault> record) {
        return record.get().getStatus().userFile.get().asPermanentFailure().map(
                failure -> previouslyFailedUploadsStatusCode.get().getOrElse(
                        () -> failure.getFailureCause().getStatusCode().getOrElse(
                                HttpStatus.SC_500_INTERNAL_SERVER_ERROR)));
    }

    public Function0<PutResult> processUploadF(
            final Record<MpfsRequestRecord.UploadToDefault> record, final HttpServletRequest req)
    {
        HttpPutRequestContext context = makePutRequestContext(req);
        boolean reset = needToResetUpload(req, !context.getContentRange().isPresent());

        Consumer<ByteArrayByteSequence> onDataReceivedCallback = Function1V.nop();
        Consumer<DataSize> onAllDataReceivedCallback = Function1V.nop();
        Option<DataSize> dataSizeO = RangedPutUtils.getUploadingContentLength(context);

        // on-the-fly digest calculation and simultaneous upload are only possible for non-multipart PUT requests
        // with defined content length
        if (dataSizeO.isPresent() && HttpMethod.PUT.name().equals(req.getMethod())
                && !MultipartUtils.isMultipartFormData(Option.ofNullable(req.getContentType())))
        {
            DataSize dataSize = dataSizeO.get();

            // on-the-fly digest calculation
            File2 digestsFile = localFileManager.allocateFile(record.get().meta.id, "on-the-fly-webdav-digest");
            OnTheFlyDigester onTheFlyDigester = new OnTheFlyDigester(digestsFile, dataSize.toBytes());

            onDataReceivedCallback = onDataReceivedCallback.andThen(onTheFlyDigester::update);
            onAllDataReceivedCallback = onAllDataReceivedCallback.andThen(totalRead -> {
                if (!dataSize.equalsTs(totalRead)) {
                    logger.warn("On the fly digests will not be used because there was a difference in file sizes");
                    return;
                }
                Digests digests = onTheFlyDigester.completeAndGetDigests();
                Instant now = new Instant();
                record.refreshAndUpdate(record.get().getStatus().userFileOnTheFlyDigests,
                        State.<Digests>initial().complete(new InstantInterval(now, now), digests));
            });

            // simultaneous upload
            if (getUidO(record).isMatch(uid -> simultaneousMulcaUploadManager.isUploadSupported(uid, dataSize))) {
                onDataReceivedCallback = onDataReceivedCallback
                        .andThen((b) -> simultaneousMulcaUploadManager.tryToSubmitUpload(record));
                onAllDataReceivedCallback = onAllDataReceivedCallback.andThen(totalRead -> {
                    if (simultaneousMulcaUploadManager.isUploadInProgress(record.get().meta.id)
                            && !State.StateType.INITIAL
                            .equals(record.refreshAndGet().getStatus().simultaneousMulcaUploadInfo.get().getType()))
                    {
                        // save the start of upload on record level
                        record.refreshAndUpdate(record.get().getStatus().simultaneousMulcaUploadInfo, State.initial());
                    }
                });
            }
        }

        Function1V<ByteArrayByteSequence> onDataReceivedCallbackF = onDataReceivedCallback::accept;
        Function1V<DataSize> onAllDataReceivedCallbackF = onAllDataReceivedCallback::accept;
        return () -> RangedPutUtils.processPutWithLock(localFileManager, record, context,
                getDiskWritePolicyOverride(record).getOrElse(diskWritePolicy), recordUpdateF(req), reset,
                onDataReceivedCallbackF, onAllDataReceivedCallbackF);
    }

    private boolean needToResetUpload(HttpServletRequest req, boolean emptyContentRange) {
        // reset uploads for iOS clients CHEMODAN-27279
        Option<YaDiskUserAgent> ua = YaDiskUserAgent.parse(HttpRequestUtils.getUserAgent(req).getOrElse(""));
        return emptyContentRange && ua.isPresent() && "iOS".equals(ua.get().os);
    }

    private Function0<RecordWrapper<?>> recordUpdateF(final HttpServletRequest req) {
        return () -> Record.cons(uploadRegistry, getRecord(req));
    }

    private MpfsRequestRecord.UploadToDefault getRecordAndLogRequest(HttpServletRequest req) {
        logger.info(HttpRequestUtils.describeRemoveSensitive(req));
        MpfsRequestRecord.UploadToDefault record = getRecord(req);

        try {
            String uidStr = record.getRequest().chemodanFile.getUidOrSpecial().toSpecialOrUidString();

            req.setAttribute(DiskLog4jRequestLog.UID_ATTRIBUTE, uidStr);
        } catch (Throwable e) {
            ExceptionUtils.throwIfUnrecoverable(e);
        }
        return record;
    }

    private MpfsRequestRecord.UploadToDefault getRecord(HttpServletRequest req) {
        final UploadRequestId id = UploadRequestId.valueOf(HttpRequestUtils.localUri(req));
        return uploadRegistry.findRecordAndDeleteIfEmpty(id).getOrThrow(() -> new RecordNotFoundException(id));
    }

    private Option<DiskWritePolicy> getDiskWritePolicyOverride(Record<MpfsRequestRecord.UploadToDefault> record) {
        Option<DataSize> uploadMaxSpeedBps = record.get().getRequest().uploadMaxSpeedBps.orElse(() ->
            getUidO(record).filterMap(userUploadSpeedLimitManager::getMaxUploadSpeed));

        return uploadMaxSpeedBps.map(diskWritePolicy::withUploadMaxSpeedBps);
    }

    private Option<PassportUid> getUidO(Record<MpfsRequestRecord.UploadToDefault> record) {
        return record.get().getRequest().chemodanFile.getUidOrSpecial().uidO().map(UidOrSpecial.Uid::getPassportUid);
    }

    private HttpPutRequestContext makePutRequestContext(HttpServletRequest req) {
        CancelableUploadInputStreamX inputStream;
        try {
            inputStream = new CancelableUploadInputStreamX(req.getInputStream());
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }

        return HttpPutRequestContextUtils.makePutRequestContext(HttpServletRequestX.wrap(req), () -> inputStream);
    }
}
