package ru.yandex.chemodan.uploader.web.data.util;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.uploader.log.Events;
import ru.yandex.chemodan.uploader.log.StageInfo;
import ru.yandex.chemodan.uploader.log.UploadStatisticLogger;
import ru.yandex.chemodan.uploader.registry.StageListenerPoolWithChildren;
import ru.yandex.chemodan.uploader.registry.record.MpfsRequestRecord;
import ru.yandex.chemodan.uploader.registry.record.Record;
import ru.yandex.chemodan.uploader.registry.record.status.MpfsRequestStatus;
import ru.yandex.commune.uploader.local.queue.LocalQueuePush;
import ru.yandex.commune.uploader.log.Event;
import ru.yandex.commune.uploader.registry.StageListenerPool;
import ru.yandex.commune.uploader.registry.StageUtils;
import ru.yandex.commune.uploader.registry.UploadRequestBaseStatus;
import ru.yandex.commune.uploader.registry.UploadRequestId;
import ru.yandex.commune.uploader.util.http.PutResult;
import ru.yandex.commune.uploader.web.data.FileAlreadyUploadedException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author vavinov
 */
public class UploadServletUtils {
    private static final Logger logger = LoggerFactory.getLogger(UploadServletUtils.class);

    // TODO handle exceptions; return status code instead of PutResult
    public static PutResult handleAndExecuteAsap(
            MpfsRequestRecord record,
            LocalQueuePush localQueuePush,
            StageListenerPoolWithChildren<Record<?>> stageListenerPool,
            Duration waitCommitFileUploadTimeout,
            Function0<PutResult> callback,
            Option<String> userAgent)
    {
        UploadRequestId id = record.meta.id;
        Instant whenStarted = new Instant();
        try {
            PutResult result = callback.apply();
            if (result == PutResult.COMPLETED) {
                result = processCompleted(id, localQueuePush, stageListenerPool, waitCommitFileUploadTimeout);
            }
            UploadStatisticLogger.logUploadEvent(record, result, userAgent);
            return result;
        } catch (FileAlreadyUploadedException e) {
            return processAlreadyUploadedRecord(id, record, stageListenerPool, waitCommitFileUploadTimeout);
        } catch (RuntimeException ex) {
            logger.warn(ex, ex);
            Events.log(
                    new StageInfo(record.getRequest().getClass().getSimpleName(), "upload"),
                    Event.failure(new Duration(whenStarted, new Instant()), ex.getMessage()));

            Option<PutResult> resultO = checkForClientFailure(ex);
            UploadStatisticLogger.logUploadEvent(record, resultO.getOrElse(PutResult.FAILED), userAgent);

            return resultO.getOrThrow(() -> ex);
        }
    }

    private static PutResult processCompleted(
            UploadRequestId id, LocalQueuePush localQueuePush,
            StageListenerPool<Record<?>> stageListenerPool, Duration waitCommitFileUploadTimeout)
    {
        return awaitForRecordFinish(
                callback -> stageListenerPool.addStageListener(id, callback),
                () -> stageListenerPool.removeStageListener(id),
                () -> localQueuePush.pushUnconditionally(id),
                waitCommitFileUploadTimeout);
    }

    private static PutResult processAlreadyUploadedRecord(UploadRequestId id, MpfsRequestRecord record,
            StageListenerPoolWithChildren<Record<?>> stageListenerPool, Duration waitCommitFileUploadTimeout)
    {
        // first, check if record was finished already
        if (isRecordFinished(record)) {
            return getResultStatusCodeIfFailure(record.getStatus())
                    .map(PutResult::byStatusCode)
                    .getOrElse(PutResult.COMPLETED);
        }

        // it wasn't, wait for that
        String childId = stageListenerPool.generateChildId();
        return awaitForRecordFinish(
                callback -> stageListenerPool.addChildListener(id, childId, callback),
                () -> stageListenerPool.removeChildListener(id, childId),
                Function0V.nop(),
                waitCommitFileUploadTimeout);
    }

    private static PutResult awaitForRecordFinish(Consumer<Function1V<Record<?>>> addListenerF,
            Runnable removeListenerF, Runnable startProcessF, Duration timeout)
    {
        AtomicInteger statusCode = new AtomicInteger(PutResult.COMPLETED.getStatusCode());
        CountDownLatch latch = new CountDownLatch(1);

        addListenerF.accept(r -> {
            if (isRecordFinished(r.get())) {
                getResultStatusCodeIfFailure(r.get().getStatus()).forEach(statusCode::set);
                latch.countDown();
            }
        });

        try {
            startProcessF.run();
            boolean wasTimeout = !latch.await(timeout.getMillis(), TimeUnit.MILLISECONDS);
            return wasTimeout ? PutResult.TIMEOUT : PutResult.byStatusCode(statusCode.get());
        } catch (InterruptedException e) {
            logger.warn(e, e);
            return PutResult.FAILED;
        } finally {
            removeListenerF.run();
        }
    }

    private static boolean isRecordFinished(MpfsRequestRecord record) {
        MpfsRequestStatus status = record.getStatus();
        Check.C.isTrue(status.isUploadingToDisk(), "Expected uploading-to-disk record, got " + record);

        return status.isCommitFileUploadFinished() || status.isFinishedWithoutFinalStages();
    }

    private static Option<Integer> getResultStatusCodeIfFailure(MpfsRequestStatus status) {
        return Option.when(status.resultWithFinalStages() == UploadRequestBaseStatus.Result.FAILED,
                () -> status.resultWithFinalPermanentFailure()
                        .filterMap(state -> state.getFailureCause().getStatusCode()
                                .filter(code -> HttpStatus.is5xx(code) || StageUtils.isFirstCallbackFailure(code)))
                        .orElse(HttpStatus.SC_503_SERVICE_UNAVAILABLE)
        );
    }

    private static Option<PutResult> checkForClientFailure(RuntimeException e) {
        if (e instanceof IllegalArgumentException || e instanceof IllegalStateException) {
            return Option.of(PutResult.BAD_REQUEST);
        }

        return Option.empty();
    }
}
