package ru.yandex.chemodan.uploader.preview;

import org.junit.Ignore;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.uploader.config.UploaderCoreContextConfigurationForTests;
import ru.yandex.commune.image.ImageFormat;
import ru.yandex.misc.image.Dimension;
import ru.yandex.misc.io.exec.ShellUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.test.Assert;

/**
 * @author akirakozov
 */
@ContextConfiguration(classes = {UploaderCoreContextConfigurationForTests.class})
public class PreviewImageManagerTest extends AbstractPreviewTest {
    // XXX after enabling tests, replace with proper dimensions
    private static final Dimension DIMENSION_UNKNOWN = Dimension.valueOf(0, 0);

    @Autowired
    private PreviewImageManager previewImageManager;

    @Test
    public void generatePreview() {
        generatePreview("vini.jpg", ImageFormat.JPEG, Dimension.valueOf(720, 544));
    }

    @Test
    public void generatePreviewPng() {
        generatePreview("cubs.png", ImageFormat.PNG, Dimension.valueOf(800, 600));
    }

    @Test
    public void generatePreviewTiff() {
        generatePreview("test.tif", ImageFormat.JPEG, Dimension.valueOf(50, 50));
    }

    @Test
    public void generatePreviewPsd() {
        generatePreview("test.psd", ImageFormat.JPEG, Dimension.valueOf(50, 50));
    }

    @Test
    public void generatePreviewIco() {
        generatePreview("test.ico", ImageFormat.JPEG, Dimension.valueOf(16, 16));
    }

    // NOT REAL TEST
    @Test
    @Ignore
    public void generateHeif() {
        PreviewInfo previewInfo = previewImageManager.generateOnePreviewForImage(
                new File2("some.heif"), Option.of("image/heif"),
                PreviewImageSize.M.getDimension(), false, "preview", Option.empty());
        Assert.equals(null, previewInfo);
    }

    @Test
    @Ignore
    public void generatePreviewSvgUbuntu() {
        generatePreview("UbuntuCoF.svg", ImageFormat.JPEG, DIMENSION_UNKNOWN);
    }

    @Ignore
    @Test
    public void generatePreviewSvgUbuntuWithXmlHeader() {
        generatePreview("UbuntuCoF-with-xml-header.svg", ImageFormat.JPEG, Dimension.valueOf(100, 100));
    }

    // http://upload.wikimedia.org/wikipedia/en/c/ce/SVG-logo.svg
    @Test
    @Ignore
    public void generatePreviewSvgLogo() {
        generatePreview("SVG-logo.svg", ImageFormat.JPEG, DIMENSION_UNKNOWN);
    }

    // http://upload.wikimedia.org/wikipedia/commons/thumb/1/15/Svg.svg/617px-Svg.svg
    @Test
    @Ignore
    public void generatePreviewSvg617px() {
        generatePreview("617px-Svg.svg", ImageFormat.JPEG, DIMENSION_UNKNOWN);
    }

    // example at http://www.w3.org/TR/SVG/struct.html
    @Test
    public void generatePreviewSvgW3cStructParentXml() {
        generatePreview("W3C-svg-struct-example.svg", ImageFormat.JPEG, Dimension.valueOf(113, 227));
    }

    @Test
    public void generatePreviewForCanonPhotoWithoutRotating() {
        File2 image = getImage("cat.jpg");
        Dimension originalDimension = Dimension.valueOf(792, 1188);

        PreviewInfo result = previewImageManager.generateOnePreviewForImage(image, Option.empty(), originalDimension,
                false, "tmpprev1.jpg", Option.empty());

        Assert.equals(originalDimension, result.getPreviewSize());
    }

    @Test
    public void generatePreviewForCr2Image() {
        File2 image = getImage("img_2635.cr2");

        PreviewInfo previewInfo = previewImageManager.generateOnePreviewForImage(image, Option.of("image/cr2"),
                PreviewImageSize.XXXL.getDimension(), false, "cr2prev.jpg", Option.empty());
        Assert.equals(Dimension.valueOf(852, 1280), previewInfo.getPreviewSize());
    }

    @Test
    public void generatePreviewForWebpImage() {
        File2 image = getImage("norway.webp");
        previewImageManager.generateOnePreviewForImage(image, Option.of("image/webp"),
                PreviewImageSize.XXXL.getDimension(), false, "webp.jpg", Option.empty());
    }

    @Test
    public void generatePreviewForPsdFile() {
        generatePreview("sign-in.psd", ImageFormat.JPEG, Dimension.valueOf(690, 387), Option.of("image/psd"));
    }

    @Test // Both of medveds are really big and is cut down till they fits 1280x1280
    public void testMedveds() {
        Dimension d = Dimension.valueOf(1698, 1131);
        Assert.lt(getPreviewSizeXXXL("medved.jpg", 1, d), getPreviewSizeXXXL("medved.jpg", 100, d));
    }

    @Test // Pasha don't need any resizing
    public void testPasha() {
        Dimension d = Dimension.valueOf(683, 1024);
        Assert.lt(getPreviewSizeXXXL("pashka.jpg", 1, d), getPreviewSizeXXXL("pashka.jpg", 100, d));

    }

    @Test
    public void testJpgInDisguise() {
        String filename = "jpg-in-disguise.HEIC";
        Dimension dimension = Dimension.valueOf(3024, 4032);
        generatePreview(filename, ImageFormat.JPEG, dimension, Option.of("image/heif"));
    }

    private long getPreviewSizeXXXL(String filename, int quality, Dimension d) {
        File2 image = getImage(filename);
        ImageFormat f = ImageFormat.JPEG;
        PreviewInfo previewInfo = previewImageManager.generateOnePreviewForImage(image, Option.empty(),
                PreviewImageSize.XXXL.getDimension(), false, "preview.jpeg", Option.of(quality));
        Assert.equals(f, previewInfo.getPreviewFormat());
        Assert.some(d, previewInfo.getOrigSizeO());
        // Using sacral knowledge of ancient Andrusha
        return image.getParent().get().child("preview.jpeg").length();
    }

    private void generatePreview(String originalFile, ImageFormat previewFormat, Dimension origSize) {
        generatePreview(originalFile, previewFormat, origSize, Option.empty());
    }

    private void generatePreview(
            String originalFile, ImageFormat previewFormat, Dimension origSize, Option<String> contentType)
    {
        File2 image = getImage(originalFile);

        PreviewInfo previewInfo = previewImageManager.generateOnePreviewForImage(image, contentType,
                PreviewImageSize.M.getDimension(), false, "preview", Option.empty());
        Assert.equals(previewFormat, previewInfo.getPreviewFormat());
        Assert.some(origSize, previewInfo.getOrigSizeO());

        String testResultXXXL = ShellUtils.executeGrabbingOutput(
                "file " + ShellUtils.quote(tmpDir + "/preview")).getStdout();
        Assert.assertContains(testResultXXXL, previewFormat + " image data");
    }
}
